/**
 * Groups Scanner Module - GraphQL API Version
 * Lấy danh sách groups qua Facebook GraphQL API thay vì DOM scraping
 * Scale được cho user có 1000+ groups
 */

import { getCookieString, getUserId } from '../../core/auth.js';
import { log, logError } from '../../core/utils.js';

/**
 * GraphQL endpoint và doc_id cho groups query
 */
const GRAPHQL_URL = 'https://www.facebook.com/api/graphql/';

// Doc ID cho GroupsCometNonMemberListContainerJoinedGroupsPaginatedGroupsListQuery
const JOINED_GROUPS_DOC_ID = '7610584142347556';

/**
 * Fetch groups list qua GraphQL API
 * @param {Object} authData - Auth data với cookies, fb_dtsg, lsd
 * @param {string} cursor - Cursor cho pagination (optional)
 * @param {number} limit - Số groups mỗi request
 * @returns {Object} { groups: Array, hasNext: boolean, endCursor: string }
 */
export async function fetchGroupsViaAPI(authData, cursor = null, limit = 100) {
  const { userId, cookies, fb_dtsg, lsd } = authData;
  
  if (!fb_dtsg || !userId) {
    throw new Error('Missing fb_dtsg or userId. Please open Facebook first.');
  }
  
  // Variables for joined groups query
  const variables = {
    "count": limit,
    "scale": 1
  };
  
  if (cursor) {
    variables.cursor = cursor;
  }
  
  // Build request body
  const formData = new URLSearchParams();
  formData.append('av', userId);
  formData.append('__user', userId);
  formData.append('fb_dtsg', fb_dtsg);
  if (lsd) formData.append('lsd', lsd);
  formData.append('jazoest', '25586');
  formData.append('__a', '1');
  formData.append('__comet_req', '15');
  formData.append('doc_id', JOINED_GROUPS_DOC_ID);
  formData.append('variables', JSON.stringify(variables));
  
  try {
    const response = await fetch(GRAPHQL_URL, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        'Accept': '*/*',
        'Origin': 'https://www.facebook.com',
        'Referer': 'https://www.facebook.com/groups/feed/',
        'X-FB-Friendly-Name': 'GroupsCometNonMemberListContainerJoinedGroupsPaginatedGroupsListQuery'
      },
      body: formData.toString(),
      credentials: 'include',
      mode: 'cors'
    });
    
    const text = await response.text();
    
    // Parse response (có thể có prefix for(;;);)
    let jsonText = text.replace(/^for \(;;\);/, '').trim();
    const data = JSON.parse(jsonText);
    
    // Extract groups from response
    return parseGroupsResponse(data);
    
  } catch (error) {
    logError('GraphQL fetch error:', error);
    throw error;
  }
}

/**
 * Parse GraphQL response để lấy groups
 * @param {Object} data - Response data
 * @returns {Object} { groups: Array, hasNext: boolean, endCursor: string }
 */
function parseGroupsResponse(data) {
  const groups = [];
  let hasNext = false;
  let endCursor = null;
  
  try {
    // Navigate to groups array in response
    // Structure varies by query, try multiple paths
    const edges = 
      data?.data?.viewer?.groups_tab?.joined_groups?.edges ||
      data?.data?.viewer?.group_tab_member_groups?.edges ||
      data?.data?.groups_tab?.tab_groups_list?.edges ||
      [];
    
    edges.forEach(edge => {
      const node = edge?.node;
      if (!node) return;
      
      let role = 'member';
      if (node.viewer_is_admin || node.is_viewer_admin) role = 'admin';
      else if (node.viewer_is_moderator || node.is_viewer_moderator) role = 'moderator';
      
      // Member count extraction with multiple fallbacks
      let memberCount = node.viewer_post_status?.member_count || node.group_membership?.member_count || null;
      
      // Fallback for formatted count text (e.g. "1.2K members", "150 members")
      if (!memberCount) {
        const formattedText = node.group_member_profiles?.formatted_count_text || '';
        if (formattedText) {
          const match = formattedText.match(/([\d.,]+)\s*(?:K|M|k|m)?/);
          if (match) {
            let val = parseFloat(match[1].replace(/,/g, ''));
            if (formattedText.toLowerCase().includes('k')) val *= 1000;
            if (formattedText.toLowerCase().includes('m')) val *= 1000000;
            memberCount = Math.floor(val);
          }
        }
      }

      const group = {
        id: node.id || node.group_id,
        name: node.name || 'Unknown Group',
        url: `/groups/${node.id || node.group_id}/`,
        avatar: node.profile_picture?.uri || node.image?.uri || null,
        memberCount: memberCount,
        role: role
      };
      
      groups.push(group);
    });
    
    // Pagination info
    const pageInfo = 
      data?.data?.viewer?.groups_tab?.joined_groups?.page_info ||
      data?.data?.viewer?.group_tab_member_groups?.page_info ||
      {};
    
    hasNext = pageInfo?.has_next_page || false;
    endCursor = pageInfo?.end_cursor || null;
    
  } catch (error) {
    logError('Error parsing groups response:', error);
  }
  
  return { groups, hasNext, endCursor };
}

/**
 * Fetch ALL groups (auto-pagination)
 * @param {Object} authData - Auth data
 * @param {Function} onProgress - Optional progress callback(currentCount, groups)
 * @returns {Array} All groups
 */
export async function fetchAllGroups(authData, onProgress = null) {
  const allGroups = [];
  let cursor = null;
  let hasNext = true;
  let page = 0;
  
  log('Starting to fetch all groups via GraphQL API...');
  
  while (hasNext) {
    try {
      page++;
      log(`Fetching page ${page}...`);
      
      const result = await fetchGroupsViaAPI(authData, cursor, 100);
      
      allGroups.push(...result.groups);
      hasNext = result.hasNext;
      cursor = result.endCursor;
      
      log(`Page ${page}: got ${result.groups.length} groups. Total: ${allGroups.length}`);
      
      if (onProgress) {
        onProgress(allGroups.length, result.groups);
      }
      
      // Small delay to avoid rate limiting
      if (hasNext) {
        await new Promise(r => setTimeout(r, 500));
      }
      
    } catch (error) {
      logError(`Error on page ${page}:`, error);
      break;
    }
  }
  
  log(`Finished! Total groups fetched: ${allGroups.length}`);
  return allGroups;
}

/**
 * Alternative: Fetch từ m.facebook.com/groups_joined
 * Đơn giản hơn, ít bị rate limit hơn
 */
export async function fetchGroupsFromMobile(authData) {
  try {
    const response = await fetch('https://m.facebook.com/groups/feed/?showgroups=1', {
      method: 'GET',
      credentials: 'include',
      headers: {
        'Accept': 'text/html',
        'User-Agent': 'Mozilla/5.0 (iPhone; CPU iPhone OS 14_0 like Mac OS X) AppleWebKit/605.1.15'
      }
    });
    
    const html = await response.text();
    return parseGroupsFromHTML(html);
    
  } catch (error) {
    logError('Mobile fetch error:', error);
    throw error;
  }
}

/**
 * Parse groups từ HTML response (m.facebook.com)
 */
function parseGroupsFromHTML(html) {
  const groups = [];
  const seenIds = new Set();
  
  // Multiple regex patterns to catch different HTML structures
  const patterns = [
    /href="\/groups\/(\d+)[^"]*"[^>]*>([^<]+)</g,
    /groups\/(\d+)(?:\/|\?)[^"]*"[^>]*>.*?<span[^>]*>([^<]+)<\/span>/gs
  ];
  
  patterns.forEach(pattern => {
    let match;
    while ((match = pattern.exec(html)) !== null) {
      const [, id, name] = match;
      if (id && !seenIds.has(id) && name && name.length > 1) {
        seenIds.add(id);
        groups.push({
          id,
          name: decodeHTMLEntities(name.trim()),
          url: `/groups/${id}/`,
          role: 'member'
        });
      }
    }
  });
  
  log(`Parsed ${groups.length} groups from mobile HTML`);
  return groups;
}

function decodeHTMLEntities(text) {
  return text
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>')
    .replace(/&quot;/g, '"')
    .replace(/&#39;/g, "'")
    .replace(/&#x27;/g, "'")
    .replace(/&#(\d+);/g, (_, num) => String.fromCharCode(num));
}

/**
 * Legacy DOM-based scanning (fallback khi API không hoạt động)
 * Chỉ dùng khi đang ở trang facebook.com/groups
 */
export function scanGroupsFromDOM() {
  const groups = [];
  const seenIds = new Set();
  
  const links = document.querySelectorAll('a[href*="/groups/"]');
  
  links.forEach(link => {
    const href = link.getAttribute('href');
    const match = href?.match(/\/groups\/(\d+)/);
    if (!match) return;
    
    const id = match[1];
    if (seenIds.has(id)) return;
    seenIds.add(id);
    
    const container = link.closest('[role="listitem"]') || 
                      link.closest('li') || 
                      link.parentElement;
    
    const nameEl = container?.querySelector('span[dir="auto"]') || link;
    const name = nameEl?.textContent?.trim() || 'Unknown';
    const avatar = container?.querySelector('img')?.src || null;
    
    groups.push({
      id,
      name,
      url: `/groups/${id}/`,
      avatar,
      role: 'member'
    });
  });
  
  log(`DOM scan: found ${groups.length} groups`);
  return groups;
}
