/**
 * Groups Poster Module - Final Version
 * Đăng bài hàng loạt lên nhiều groups với text + ảnh
 * Based on captured Facebook GraphQL payload
 */

import { getCookieString } from '../../core/auth.js';
import { delay, log, logError, randomNumber } from '../../core/utils.js';
import { getSettings } from '../../core/storage.js';

// ==================== Constants ====================

const UPLOAD_URL = 'https://upload.facebook.com/ajax/react_composer/attachments/photo/upload';
const GRAPHQL_URL = 'https://www.facebook.com/api/graphql/';
const DOC_ID = '25773390049008020';

// ==================== Upload Photo ====================

/**
 * Upload ảnh lên Facebook
 * @param {File} imageFile - File từ input type="file"
 * @param {Object} params - { userId, fbDtsg, lsd }
 * @returns {Promise<string|null>} photoID nếu thành công
 */
export async function uploadPhoto(imageFile, params) {
  const { userId, fbDtsg, lsd } = params;
  
  if (!imageFile) return null;
  
  try {
    log(`Uploading photo: ${imageFile.name}...`);
    
    const formData = new FormData();
    formData.append('source', '8');
    formData.append('profile_id', userId);
    formData.append('waterfallxapp', 'comet');
    formData.append('farr', imageFile, imageFile.name);
    formData.append('upload_id', `jsc_c_${Date.now()}`);
    formData.append('fb_dtsg', fbDtsg);
    formData.append('jazoest', '25296');
    if (lsd) formData.append('lsd', lsd);
    
    const url = `${UPLOAD_URL}?av=${userId}&__user=${userId}&__a=1`;
    const cookieString = await getCookieString();
    
    const response = await fetch(url, {
      method: 'POST',
      body: formData,
      credentials: 'include',
      mode: 'cors',
      headers: {
        'Cookie': cookieString,
        'Accept': '*/*',
        'Sec-Fetch-Dest': 'empty',
        'Sec-Fetch-Mode': 'cors',
        'Sec-Fetch-Site': 'same-origin'
      }
    });
    
    const text = await response.text();
    
    // Parse response: bỏ prefix 'for (;;);' và parse JSON
    let jsonText = text.replace(/^for \(;;\);/, '').trim();
    
    if (jsonText.startsWith('{')) {
      const json = JSON.parse(jsonText);
      const photoId = json?.payload?.photoID || json?.payload?.photo_id;
      
      if (photoId) {
        log(`✓ Photo uploaded: ${photoId}`);
        return photoId;
      } else {
        logError('No photoID in response:', json);
      }
    } else {
      logError('Invalid response format:', text.substring(0, 200));
    }
    
    return null;
    
  } catch (error) {
    logError('Upload failed:', error);
    return null;
  }
}

// ==================== Post to Group ====================

/**
 * Đăng bài lên một group
 * @param {Object} params
 * @returns {Promise<Object>}
 */
export async function postToGroup(params) {
  const {
    groupId,
    message,
    userId,
    fbDtsg,
    lsd = '',
    photoIds = []
  } = params;
  
  try {
    log(`Posting to group ${groupId}...`);
    
    // Build attachments
    const attachments = photoIds.map(id => ({ photo: { id } }));
    
    // Generate IDs
    const clientMutationId = Math.floor(Math.random() * 1000).toString();
    const composerSessionId = typeof crypto !== 'undefined' && crypto.randomUUID 
      ? crypto.randomUUID() 
      : `${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
    
    // Build variables — matched to real captured Facebook payload
    const inputObj = {
      composer_entry_point: "inline_composer",
      composer_source_surface: "group",
      composer_type: "group",
      logging: { composer_session_id: composerSessionId },
      source: "WWW",
      message: { ranges: [], text: message },
      with_tags_ids: null,
      inline_activities: [],
      text_format_preset_id: "0",
      group_flair: { flair_id: null },
      composed_text: {
        block_data: ["{}"],
        block_depths: [0],
        block_types: [0],
        blocks: [message],
        entities: ["[]"],
        entity_map: "{}",
        inline_styles: ["[]"]
      },
      navigation_data: { 
        attribution_id_v2: `CometGroupDiscussionRoot.react,comet.group,via_cold_start,${Date.now()},763346,2361831622,,` 
      },
      tracking: [null],
      event_share_metadata: { surface: "newsfeed" },
      audience: { to_id: groupId },
      actor_id: userId,
      client_mutation_id: clientMutationId
    };
    
    // Only add attachments if we have photos
    if (attachments.length > 0) {
      inputObj.attachments = attachments;
    }
    
    const variables = {
      input: inputObj,
      feedLocation: "GROUP",
      feedbackSource: 0,
      focusCommentID: null,
      gridMediaWidth: null,
      groupID: null,
      scale: 1,
      privacySelectorRenderLocation: "COMET_STREAM",
      checkPhotosToReelsUpsellEligibility: false,
      renderLocation: "group",
      useDefaultActor: false,
      inviteShortLinkKey: null,
      isFeed: false,
      isFundraiser: false,
      isFunFactPost: false,
      isGroup: true,
      isEvent: false,
      isTimeline: false,
      isSocialLearning: false,
      isPageNewsFeed: false,
      isProfileReviews: false,
      isWorkSharedDraft: false,
      hashtag: null,
      canUserManageOffers: false
    };
    
    // Build FormData body
    const bodyParams = new URLSearchParams();
    bodyParams.append('av', userId);
    bodyParams.append('__user', userId);
    bodyParams.append('__a', '1');
    bodyParams.append('fb_dtsg', fbDtsg);
    bodyParams.append('jazoest', '25296');
    if (lsd) bodyParams.append('lsd', lsd);
    bodyParams.append('__comet_req', '15');
    bodyParams.append('fb_api_caller_class', 'RelayModern');
    bodyParams.append('fb_api_req_friendly_name', 'ComposerStoryCreateMutation');
    bodyParams.append('server_timestamps', 'true');
    bodyParams.append('doc_id', DOC_ID);
    bodyParams.append('variables', JSON.stringify(variables));
    
    const cookieString = await getCookieString();
    
    const response = await fetch(GRAPHQL_URL, {
      method: 'POST',
      credentials: 'include',
      mode: 'cors',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        'Cookie': cookieString,
        'Accept': '*/*',
        'X-FB-Friendly-Name': 'ComposerStoryCreateMutation',
        'X-FB-LSD': lsd || '',
        'Sec-Fetch-Dest': 'empty',
        'Sec-Fetch-Mode': 'cors',
        'Sec-Fetch-Site': 'same-origin'
      },
      body: bodyParams.toString()
    });
    
    const text = await response.text();
    log(`[DEBUG] HTTP ${response.status} for group ${groupId}, response (first 500 chars):`, text.substring(0, 500));
    const result = parseResponse(text);
    
    // Check for success
    const postId = result?.data?.story_create?.story?.legacy_story_hideable_id ||
                   result?.data?.story_create?.post_id;
    
    if (postId) {
      log(`✓ Posted successfully: ${postId}`);
      return { success: true, postId, groupId };
    }
    
    // Check for error — include raw snippet for debugging
    const errorMsg = result?.errors?.[0]?.description || 
                     result?.errors?.[0]?.message || 
                     result?.error?.message ||
                     (text.includes('error') ? `FB error: ${text.substring(0, 200)}` : 'No post ID');
    logError(`✗ Failed: ${errorMsg}`);
    return { success: false, error: errorMsg, groupId };
    
  } catch (error) {
    logError(`✗ Exception: ${error.message}`);
    return { success: false, error: error.message, groupId };
  }
}

// ==================== Bulk Post ====================

/**
 * Đăng bài hàng loạt lên nhiều groups
 */
export async function bulkPost(groupIds, message, fbDtsg, userId, options = {}) {
  const { 
    minDelay = 5000, 
    maxDelay = 10000, 
    imageData = null, 
    isAnonymous = false,
    lsd = '',
    onProgress = () => {} 
  } = options;
  
  if (!userId || !fbDtsg) {
    throw new Error('Missing userId or fb_dtsg');
  }
  
  const results = {
    total: groupIds.length,
    success: 0,
    failed: 0,
    posts: [],
    logs: []
  };
  
  // Upload photo ONCE nếu có
  let photoIds = [];
  if (imageData) {
    onProgress({ status: 'uploading', message: 'Đang upload ảnh...' });
    
    // Convert base64 to File if needed
    let imageFile = imageData;
    if (typeof imageData === 'string' && imageData.startsWith('data:')) {
      const blob = await (await fetch(imageData)).blob();
      imageFile = new File([blob], 'photo.jpg', { type: blob.type });
    }
    
    const photoId = await uploadPhoto(imageFile, { userId, fbDtsg, lsd });
    
    if (photoId) {
      photoIds = [photoId];
      results.logs.push(`✓ Upload ảnh thành công: ${photoId}`);
    } else {
      // Don't fail the whole bulk if photo fails? User might want text-only fallback?
      // For now, let's keep it strict if photo was provided.
      results.logs.push(`✗ Upload ảnh thất bại`);
    }
  }
  
  // Post to each group
  for (let i = 0; i < groupIds.length; i++) {
    const groupId = groupIds[i];
    
    onProgress({
      current: i + 1,
      total: groupIds.length,
      groupId,
      status: 'posting'
    });
    
    const result = await postToGroup({
      groupId,
      message,
      userId,
      fbDtsg,
      lsd,
      photoIds,
      isAnonymous
    });
    
    results.posts.push(result);
    
    if (result.success) {
      results.success++;
      results.logs.push(`✓ ${groupId}: ${result.postId}`);
    } else {
      results.failed++;
      results.logs.push(`✗ ${groupId}: ${result.error}`);
    }
    
    onProgress({
      current: i + 1,
      total: groupIds.length,
      groupId,
      status: result.success ? 'success' : 'failed',
      result
    });
    
    // Random delay
    if (i < groupIds.length - 1) {
      const waitTime = Math.floor(Math.random() * (maxDelay - minDelay + 1) + minDelay);
      log(`Waiting ${waitTime / 1000}s...`);
      
      onProgress({
        current: i + 1,
        total: groupIds.length,
        status: 'waiting',
        waitTime: waitTime / 1000
      });
      
      await delay(waitTime);
    }
  }
  
  log(`Done: ${results.success}/${results.total} success`);
  return {
    success: true,
    summary: results,
    results: results.posts
  };
}

// ==================== Bulk Post Multi-Content ====================

/**
 * Đăng bài với nội dung khác nhau cho mỗi group
 * @param {Array<{groupId: string, message: string}>} posts - Array of {groupId, message}
 * @param {string} fbDtsg
 * @param {string} userId
 * @param {Object} options
 */
export async function bulkPostMulti(posts, fbDtsg, userId, options = {}) {
  const {
    minDelay = 5000,
    maxDelay = 10000,
    imageData = null,
    isAnonymous = false,
    lsd = '',
    onProgress = () => {}
  } = options;

  if (!userId || !fbDtsg) throw new Error('Missing userId or fb_dtsg');

  const results = {
    total: posts.length,
    success: 0,
    failed: 0,
    posts: [],
    logs: []
  };

  // Upload photo ONCE if provided
  let photoIds = [];
  if (imageData) {
    onProgress({ status: 'uploading', message: 'Đang upload ảnh...' });
    let imageFile = imageData;
    if (typeof imageData === 'string' && imageData.startsWith('data:')) {
      const blob = await (await fetch(imageData)).blob();
      imageFile = new File([blob], 'photo.jpg', { type: blob.type });
    }
    const photoId = await uploadPhoto(imageFile, { userId, fbDtsg, lsd });
    if (photoId) {
      photoIds = [photoId];
      results.logs.push(`✓ Upload ảnh thành công: ${photoId}`);
    } else {
      results.logs.push(`✗ Upload ảnh thất bại`);
    }
  }

  for (let i = 0; i < posts.length; i++) {
    const { groupId, message } = posts[i];

    onProgress({ current: i + 1, total: posts.length, groupId, status: 'posting' });

    const result = await postToGroup({
      groupId, message, userId, fbDtsg, lsd, photoIds, isAnonymous
    });

    results.posts.push(result);
    if (result.success) {
      results.success++;
      results.logs.push(`✓ ${groupId}: ${result.postId}`);
    } else {
      results.failed++;
      results.logs.push(`✗ ${groupId}: ${result.error}`);
    }

    onProgress({
      current: i + 1, total: posts.length, groupId,
      status: result.success ? 'success' : 'failed', result
    });

    if (i < posts.length - 1) {
      const waitTime = Math.floor(Math.random() * (maxDelay - minDelay + 1) + minDelay);
      log(`Waiting ${waitTime / 1000}s...`);
      onProgress({ current: i + 1, total: posts.length, status: 'waiting', waitTime: waitTime / 1000 });
      await delay(waitTime);
    }
  }

  log(`Done: ${results.success}/${results.total} success`);
  return { success: true, summary: results, results: results.posts };
}

// ==================== Helpers ====================

function parseResponse(text) {
  try {
    let cleaned = text.replace(/^for \(;;\);/, '').trim();
    if (cleaned.startsWith('{')) return JSON.parse(cleaned);
    const lines = cleaned.split('\n').filter(line => line.trim());
    if (lines.length > 0 && lines[0].startsWith('{')) return JSON.parse(lines[0]);
  } catch (error) {
    logError('Parse error:', error);
  }
  return null;
}

export { DOC_ID };
