/**
 * Groups Moderation Module
 * Bulk decline, pending remover, reported content management
 * Based on captured Facebook GraphQL APIs
 */

import { getCookieString } from '../../core/auth.js';
import { delay, log, logError } from '../../core/utils.js';

// ==================== Constants ====================

const GRAPHQL_URL = 'https://www.facebook.com/api/graphql/';

// Doc IDs from capture analysis
const DOC_IDS = {
  // Decline mutations
  DECLINE_PENDING_STORY: '24257178983980732',
  DECLINE_MODMIN_REVIEW: '30216535437945304',
  
  // Pending posts queries  
  PENDING_POSTS_FEED: '25681364944878024',
  PENDING_POSTS_SECTION: '25504074775955204',
  PENDING_POSTS_ROOT: '24640081202263810',
  
  // Reported content
  REPORTED_CONTENT: '33644700051810970',
  KEYWORD_ALERTED: '26610054021930816',
  
  // Modmin review
  MODMIN_REVIEW_ROOT: '25859218577065816'
};

// ==================== GraphQL Request Helper ====================

async function graphqlRequest(docId, variables, params = {}) {
  const { userId, fbDtsg, lsd = '' } = params;
  
  const bodyParams = new URLSearchParams();
  bodyParams.append('av', userId);
  bodyParams.append('__user', userId);
  bodyParams.append('__a', '1');
  bodyParams.append('fb_dtsg', fbDtsg);
  bodyParams.append('jazoest', '25296');
  if (lsd) bodyParams.append('lsd', lsd);
  bodyParams.append('__comet_req', '15');
  bodyParams.append('fb_api_caller_class', 'RelayModern');
  bodyParams.append('server_timestamps', 'true');
  bodyParams.append('doc_id', docId);
  bodyParams.append('variables', JSON.stringify(variables));
  
  const cookieString = await getCookieString();
  
  const response = await fetch(GRAPHQL_URL, {
    method: 'POST',
    credentials: 'include',
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded',
      'Cookie': cookieString,
      'Accept': '*/*',
      'Sec-Fetch-Dest': 'empty',
      'Sec-Fetch-Mode': 'cors',
      'Sec-Fetch-Site': 'same-origin'
    },
    body: bodyParams.toString()
  });
  
  const text = await response.text();
  return parseResponse(text);
}

function parseResponse(text) {
  try {
    let cleaned = text.replace(/^for \(;;\\);/, '').trim();
    if (cleaned.startsWith('{')) {
      return JSON.parse(cleaned);
    }
    const lines = cleaned.split('\n').filter(line => line.trim());
    if (lines.length > 0 && lines[0].startsWith('{')) {
      return JSON.parse(lines[0]);
    }
  } catch (error) {
    logError('Parse error:', error);
  }
  return null;
}

// ==================== Decline Pending Post ====================

/**
 * Decline a single pending story
 * @param {Object} params
 * @returns {Promise<Object>}
 */
export async function declinePendingStory(params) {
  const { storyId, groupId, userId, fbDtsg, lsd } = params;
  
  try {
    log(`Declining story ${storyId}...`);
    
    const variables = {
      input: {
        story_id: storyId,
        group_id: groupId,
        actor_id: userId,
        client_mutation_id: Math.floor(Math.random() * 1000).toString()
      }
    };
    
    const result = await graphqlRequest(
      DOC_IDS.DECLINE_PENDING_STORY,
      variables,
      { userId, fbDtsg, lsd }
    );
    
    // Check success
    if (result?.data?.group_pending_story_decline) {
      log(`✓ Declined: ${storyId}`);
      return { success: true, storyId };
    }
    
    const errorMsg = result?.errors?.[0]?.message || 'Unknown error';
    return { success: false, storyId, error: errorMsg };
    
  } catch (error) {
    logError(`✗ Decline failed: ${error.message}`);
    return { success: false, storyId, error: error.message };
  }
}

/**
 * Decline using modmin review folder mutation (alternative)
 */
export async function declineModminReviewContent(params) {
  const { storyId, groupId, userId, fbDtsg, lsd, reason = 'spam' } = params;
  
  try {
    const variables = {
      input: {
        story_id: storyId,
        group_id: groupId,
        decline_reason: reason,
        actor_id: userId,
        client_mutation_id: Math.floor(Math.random() * 1000).toString()
      }
    };
    
    const result = await graphqlRequest(
      DOC_IDS.DECLINE_MODMIN_REVIEW,
      variables,
      { userId, fbDtsg, lsd }
    );
    
    if (result?.data) {
      log(`✓ Declined (modmin): ${storyId}`);
      return { success: true, storyId };
    }
    
    return { success: false, storyId, error: 'Failed' };
    
  } catch (error) {
    return { success: false, storyId, error: error.message };
  }
}

// ==================== Bulk Decline ====================

/**
 * Bulk decline multiple pending stories
 * @param {Array<Object>} stories - Array of { storyId, groupId }
 * @param {Object} params - { userId, fbDtsg, lsd }
 * @param {Object} options - { delayMs, onProgress }
 * @returns {Promise<Object>}
 */
export async function bulkDecline(stories, params, options = {}) {
  const { userId, fbDtsg, lsd } = params;
  const { delayMs = 500, onProgress = () => {} } = options;
  
  const results = {
    total: stories.length,
    success: 0,
    failed: 0,
    items: []
  };
  
  for (let i = 0; i < stories.length; i++) {
    const story = stories[i];
    
    onProgress({
      current: i + 1,
      total: stories.length,
      storyId: story.storyId,
      status: 'declining'
    });
    
    const result = await declinePendingStory({
      storyId: story.storyId,
      groupId: story.groupId,
      userId,
      fbDtsg,
      lsd
    });
    
    results.items.push(result);
    
    if (result.success) {
      results.success++;
    } else {
      results.failed++;
    }
    
    onProgress({
      current: i + 1,
      total: stories.length,
      storyId: story.storyId,
      status: result.success ? 'success' : 'failed',
      result
    });
    
    // Rate limiting
    if (i < stories.length - 1 && delayMs > 0) {
      await delay(delayMs);
    }
  }
  
  log(`Bulk decline done: ${results.success}/${results.total}`);
  return results;
}

// ==================== Fetch Pending Posts ====================

/**
 * Fetch pending posts for a group
 * @param {Object} params
 * @returns {Promise<Array>}
 */
export async function fetchPendingPosts(params) {
  const { groupId, userId, fbDtsg, lsd, cursor = null, limit = 10 } = params;
  
  try {
    log(`Fetching pending posts for group ${groupId}...`);
    
    const variables = {
      groupID: groupId,
      count: limit,
      cursor: cursor,
      scale: 1
    };
    
    const result = await graphqlRequest(
      DOC_IDS.PENDING_POSTS_FEED,
      variables,
      { userId, fbDtsg, lsd }
    );
    
    // Extract stories from response
    const edges = result?.data?.node?.pending_stories?.edges || 
                  result?.data?.group?.pending_stories?.edges || [];
    
    const stories = edges.map(edge => ({
      storyId: edge.node?.id || edge.node?.legacy_story_id,
      authorName: edge.node?.actors?.[0]?.name,
      authorId: edge.node?.actors?.[0]?.id,
      message: edge.node?.message?.text || edge.node?.comet_sections?.content?.story?.message?.text,
      createdTime: edge.node?.creation_time
    }));
    
    const pageInfo = result?.data?.node?.pending_stories?.page_info || 
                     result?.data?.group?.pending_stories?.page_info || {};
    
    return {
      stories,
      hasNextPage: pageInfo.has_next_page,
      endCursor: pageInfo.end_cursor
    };
    
  } catch (error) {
    logError(`Fetch pending failed: ${error.message}`);
    return { stories: [], hasNextPage: false, endCursor: null };
  }
}

/**
 * Fetch ALL pending posts (paginated)
 */
export async function fetchAllPendingPosts(params) {
  const { groupId, userId, fbDtsg, lsd, maxPages = 10, onProgress = () => {} } = params;
  
  let allStories = [];
  let cursor = null;
  let page = 0;
  
  while (page < maxPages) {
    page++;
    onProgress({ page, totalFetched: allStories.length, status: 'fetching' });
    
    const result = await fetchPendingPosts({
      groupId,
      userId,
      fbDtsg,
      lsd,
      cursor,
      limit: 20
    });
    
    allStories = allStories.concat(result.stories);
    
    if (!result.hasNextPage) break;
    cursor = result.endCursor;
    
    await delay(500); // Rate limit between pages
  }
  
  onProgress({ page, totalFetched: allStories.length, status: 'done' });
  return allStories;
}

// ==================== Pending Posts Remover ====================

/**
 * Remove all pending posts from a group
 * @param {Object} params
 * @returns {Promise<Object>}
 */
export async function removeAllPendingPosts(params) {
  const { groupId, userId, fbDtsg, lsd, onProgress = () => {} } = params;
  
  // Step 1: Fetch all pending
  onProgress({ phase: 'fetching', message: 'Đang tải danh sách pending...' });
  
  const stories = await fetchAllPendingPosts({
    groupId,
    userId,
    fbDtsg,
    lsd,
    onProgress: (p) => onProgress({ ...p, phase: 'fetching' })
  });
  
  if (stories.length === 0) {
    return { success: true, message: 'Không có pending posts', declined: 0 };
  }
  
  // Step 2: Bulk decline
  onProgress({ phase: 'declining', message: `Đang từ chối ${stories.length} posts...` });
  
  const storiesToDecline = stories.map(s => ({
    storyId: s.storyId,
    groupId
  }));
  
  const results = await bulkDecline(storiesToDecline, { userId, fbDtsg, lsd }, {
    delayMs: 500,
    onProgress: (p) => onProgress({ ...p, phase: 'declining' })
  });
  
  return {
    success: true,
    message: `Đã từ chối ${results.success}/${results.total} posts`,
    declined: results.success,
    failed: results.failed
  };
}

// ==================== Reported Content ====================

/**
 * Fetch reported content stories
 */
export async function fetchReportedContent(params) {
  const { groupId, userId, fbDtsg, lsd, cursor = null } = params;
  
  try {
    const variables = {
      groupID: groupId,
      count: 20,
      cursor: cursor,
      scale: 1
    };
    
    const result = await graphqlRequest(
      DOC_IDS.REPORTED_CONTENT,
      variables,
      { userId, fbDtsg, lsd }
    );
    
    const edges = result?.data?.node?.reported_stories?.edges || [];
    
    const stories = edges.map(edge => ({
      storyId: edge.node?.id,
      authorName: edge.node?.actors?.[0]?.name,
      reportReason: edge.node?.admin_report_info?.report_reason,
      reportCount: edge.node?.admin_report_info?.report_count,
      message: edge.node?.message?.text
    }));
    
    return { stories };
    
  } catch (error) {
    logError(`Fetch reported failed: ${error.message}`);
    return { stories: [] };
  }
}

// ==================== Export ====================

export { DOC_IDS };
