/**
 * Groups List Extractor Module
 * Extract groups you're in and groups you manage
 * Based on captured Facebook GraphQL APIs
 */

import { getCookieString } from '../../core/auth.js';
import { delay, log, logError } from '../../core/utils.js';

// ==================== Constants ====================

const GRAPHQL_URL = 'https://www.facebook.com/api/graphql/';

// Doc IDs from capture
const DOC_IDS = {
  YOUR_GROUPS: '9658982227546884',      // GroupsLeftRailYourGroupsPaginatedQuery
  GROUPS_YOU_MANAGE: '10000015690112056' // GroupsLeftRailGroupsYouManagePaginatedQuery
};

// ==================== GraphQL Helper ====================

async function graphqlRequest(docId, variables, params = {}) {
  const { userId, fbDtsg, lsd = '' } = params;
  
  const bodyParams = new URLSearchParams();
  bodyParams.append('av', userId);
  bodyParams.append('__user', userId);
  bodyParams.append('__a', '1');
  bodyParams.append('fb_dtsg', fbDtsg);
  bodyParams.append('jazoest', '25296');
  if (lsd) bodyParams.append('lsd', lsd);
  bodyParams.append('__comet_req', '15');
  bodyParams.append('fb_api_caller_class', 'RelayModern');
  bodyParams.append('server_timestamps', 'true');
  bodyParams.append('doc_id', docId);
  bodyParams.append('variables', JSON.stringify(variables));
  
  const cookieString = await getCookieString();
  
  const response = await fetch(GRAPHQL_URL, {
    method: 'POST',
    credentials: 'include',
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded',
      'Cookie': cookieString,
      'Accept': '*/*',
      'Sec-Fetch-Dest': 'empty',
      'Sec-Fetch-Mode': 'cors',
      'Sec-Fetch-Site': 'same-origin'
    },
    body: bodyParams.toString()
  });
  
  const text = await response.text();
  return parseResponse(text);
}

function parseResponse(text) {
  try {
    let cleaned = text.replace(/^for \(;;\\);/, '').trim();
    if (cleaned.startsWith('{')) {
      return JSON.parse(cleaned);
    }
    const lines = cleaned.split('\n').filter(line => line.trim());
    if (lines.length > 0 && lines[0].startsWith('{')) {
      return JSON.parse(lines[0]);
    }
  } catch (error) {
    logError('Parse error:', error);
  }
  return null;
}

// ==================== Fetch Your Groups ====================

/**
 * Fetch groups you're a member of
 * @param {Object} params
 * @returns {Promise<Object>}
 */
export async function fetchYourGroups(params) {
  const { userId, fbDtsg, lsd, cursor = null, limit = 20 } = params;
  
  try {
    log('Fetching your groups...');
    
    const variables = {
      count: limit,
      cursor: cursor,
      scale: 1
    };
    
    const result = await graphqlRequest(
      DOC_IDS.YOUR_GROUPS,
      variables,
      { userId, fbDtsg, lsd }
    );
    
    // Extract groups from response
    const edges = result?.data?.viewer?.groups_tab?.tab_groups_list?.edges || [];
    
    const groups = edges.map(edge => ({
      id: edge.node?.id,
      name: edge.node?.name,
      memberCount: edge.node?.group_member_count,
      url: edge.node?.url,
      coverPhoto: edge.node?.cover_photo?.photo?.image?.uri,
      isAdmin: edge.node?.viewer_admin_type === 'ADMIN',
      isModerator: edge.node?.viewer_admin_type === 'MODERATOR'
    }));
    
    const pageInfo = result?.data?.viewer?.groups_tab?.tab_groups_list?.page_info || {};
    
    return {
      groups,
      hasNextPage: pageInfo.has_next_page,
      endCursor: pageInfo.end_cursor
    };
    
  } catch (error) {
    logError(`Fetch groups failed: ${error.message}`);
    return { groups: [], hasNextPage: false, endCursor: null };
  }
}

/**
 * Fetch groups you manage (admin/moderator)
 */
export async function fetchGroupsYouManage(params) {
  const { userId, fbDtsg, lsd, cursor = null, limit = 20 } = params;
  
  try {
    log('Fetching groups you manage...');
    
    const variables = {
      count: limit,
      cursor: cursor,
      scale: 1
    };
    
    const result = await graphqlRequest(
      DOC_IDS.GROUPS_YOU_MANAGE,
      variables,
      { userId, fbDtsg, lsd }
    );
    
    const edges = result?.data?.viewer?.groups_tab?.groups_you_manage?.edges || [];
    
    const groups = edges.map(edge => ({
      id: edge.node?.id,
      name: edge.node?.name,
      memberCount: edge.node?.group_member_count,
      url: edge.node?.url,
      pendingCount: edge.node?.pending_member_count,
      isAdmin: true
    }));
    
    const pageInfo = result?.data?.viewer?.groups_tab?.groups_you_manage?.page_info || {};
    
    return {
      groups,
      hasNextPage: pageInfo.has_next_page,
      endCursor: pageInfo.end_cursor
    };
    
  } catch (error) {
    logError(`Fetch managed groups failed: ${error.message}`);
    return { groups: [], hasNextPage: false, endCursor: null };
  }
}

// ==================== Fetch All Groups ====================

/**
 * Fetch all groups (paginated)
 * @param {string} type - 'all' | 'managed'
 * @param {Object} params
 * @returns {Promise<Array>}
 */
export async function fetchAllGroups(type, params) {
  const { userId, fbDtsg, lsd, maxPages = 10, onProgress = () => {} } = params;
  
  const fetchFn = type === 'managed' ? fetchGroupsYouManage : fetchYourGroups;
  
  let allGroups = [];
  let cursor = null;
  let page = 0;
  
  while (page < maxPages) {
    page++;
    onProgress({ page, totalFetched: allGroups.length, status: 'fetching' });
    
    const result = await fetchFn({
      userId,
      fbDtsg,
      lsd,
      cursor,
      limit: 30
    });
    
    allGroups = allGroups.concat(result.groups);
    
    if (!result.hasNextPage) break;
    cursor = result.endCursor;
    
    await delay(500);
  }
  
  onProgress({ page, totalFetched: allGroups.length, status: 'done' });
  return allGroups;
}

// ==================== Export Groups ====================

/**
 * Export groups to various formats
 */
export function exportGroupsAsCSV(groups) {
  const headers = ['ID', 'Name', 'Member Count', 'URL', 'Is Admin'];
  const rows = groups.map(g => [
    g.id,
    `"${(g.name || '').replace(/"/g, '""')}"`,
    g.memberCount || '',
    g.url || '',
    g.isAdmin ? 'Yes' : 'No'
  ]);
  
  return [headers.join(','), ...rows.map(r => r.join(','))].join('\n');
}

export function exportGroupsAsJSON(groups) {
  return JSON.stringify(groups, null, 2);
}

/**
 * Extract group IDs only
 */
export function extractGroupIds(groups) {
  return groups.map(g => g.id).filter(Boolean);
}

// ==================== Export ====================

export { DOC_IDS };
