/**
 * Core Utils Module - Common utilities
 */

/**
 * Tạo delay ngẫu nhiên trong khoảng min-max giây
 * @param {number} minSeconds - Minimum seconds (default: 15)
 * @param {number} maxSeconds - Maximum seconds (default: 90)
 * @returns {Promise<void>}
 */
export function randomDelay(minSeconds = 15, maxSeconds = 90) {
  const delayMs = Math.floor(
    Math.random() * (maxSeconds - minSeconds + 1) + minSeconds
  ) * 1000;
  
  console.log(`[Utils] Waiting ${delayMs / 1000}s...`);
  return new Promise(resolve => setTimeout(resolve, delayMs));
}

/**
 * Delay cố định
 * @param {number} ms - Milliseconds
 * @returns {Promise<void>}
 */
export function delay(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

/**
 * Format timestamp thành readable string
 * @param {number} timestamp
 * @returns {string}
 */
export function formatTime(timestamp = Date.now()) {
  const date = new Date(timestamp);
  return date.toLocaleString('vi-VN', {
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit'
  });
}

/**
 * Format date đầy đủ
 * @param {number} timestamp
 * @returns {string}
 */
export function formatDate(timestamp = Date.now()) {
  const date = new Date(timestamp);
  return date.toLocaleString('vi-VN', {
    year: 'numeric',
    month: '2-digit',
    day: '2-digit',
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit'
  });
}

/**
 * Tạo random string (cho mutation id, etc.)
 * @param {number} length
 * @returns {string}
 */
export function randomString(length = 10) {
  const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
  let result = '';
  for (let i = 0; i < length; i++) {
    result += chars.charAt(Math.floor(Math.random() * chars.length));
  }
  return result;
}

/**
 * Tạo random number trong khoảng
 * @param {number} min
 * @param {number} max
 * @returns {number}
 */
export function randomNumber(min, max) {
  return Math.floor(Math.random() * (max - min + 1)) + min;
}

/**
 * Parse JSON an toàn
 * @param {string} str
 * @param {any} defaultValue
 * @returns {any}
 */
export function safeJsonParse(str, defaultValue = null) {
  try {
    return JSON.parse(str);
  } catch {
    return defaultValue;
  }
}

/**
 * Truncate text với ellipsis
 * @param {string} text
 * @param {number} maxLength
 * @returns {string}
 */
export function truncate(text, maxLength = 50) {
  if (!text) return '';
  if (text.length <= maxLength) return text;
  return text.substring(0, maxLength) + '...';
}

/**
 * Extract group ID từ URL
 * @param {string} url
 * @returns {string|null}
 */
export function extractGroupId(url) {
  if (!url) return null;
  
  // Pattern 1: facebook.com/groups/123456789
  const numericMatch = url.match(/\/groups\/(\d+)/);
  if (numericMatch) return numericMatch[1];
  
  // Pattern 2: facebook.com/groups/groupname (permalink)
  const nameMatch = url.match(/\/groups\/([^/?]+)/);
  if (nameMatch) return nameMatch[1];
  
  return null;
}

/**
 * Escape HTML để prevent XSS
 * @param {string} str
 * @returns {string}
 */
export function escapeHtml(str) {
  if (!str) return '';
  const div = document.createElement('div');
  div.textContent = str;
  return div.innerHTML;
}

/**
 * Log với timestamp prefix
 * @param  {...any} args
 */
export function log(...args) {
  console.log(`[${formatTime()}]`, ...args);
}

/**
 * Error log với timestamp
 * @param  {...any} args
 */
export function logError(...args) {
  console.error(`[${formatTime()}] ERROR:`, ...args);
}
