/**
 * Core Storage Module - Wrapper cho chrome.storage
 * Lưu trữ data: groups list, auth data, settings, etc.
 */

const STORAGE_KEYS = {
  AUTH_DATA: 'fb_auth_data',
  GROUPS_LIST: 'fb_groups_list',
  SETTINGS: 'fb_settings',
  POST_HISTORY: 'fb_post_history'
};

/**
 * Lưu data vào chrome.storage.local
 * @param {string} key - Storage key
 * @param {any} value - Data to store
 * @returns {Promise<void>}
 */
export async function setItem(key, value) {
  return new Promise((resolve, reject) => {
    chrome.storage.local.set({ [key]: value }, () => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else {
        resolve();
      }
    });
  });
}

/**
 * Lấy data từ chrome.storage.local
 * @param {string} key - Storage key
 * @returns {Promise<any>}
 */
export async function getItem(key) {
  return new Promise((resolve, reject) => {
    chrome.storage.local.get([key], (result) => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else {
        resolve(result[key]);
      }
    });
  });
}

/**
 * Xóa item từ storage
 * @param {string} key - Storage key
 * @returns {Promise<void>}
 */
export async function removeItem(key) {
  return new Promise((resolve, reject) => {
    chrome.storage.local.remove([key], () => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else {
        resolve();
      }
    });
  });
}

/**
 * Clear toàn bộ storage
 * @returns {Promise<void>}
 */
export async function clearAll() {
  return new Promise((resolve, reject) => {
    chrome.storage.local.clear(() => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else {
        resolve();
      }
    });
  });
}

// ==================== Specific Storage Functions ====================

/**
 * Lưu auth data
 * @param {Object} authData
 */
export async function saveAuthData(authData) {
  await setItem(STORAGE_KEYS.AUTH_DATA, authData);
}

/**
 * Lấy auth data
 * @returns {Promise<Object|null>}
 */
export async function getAuthData() {
  return await getItem(STORAGE_KEYS.AUTH_DATA);
}

/**
 * Lưu danh sách groups
 * @param {Array} groups
 */
export async function saveGroupsList(groups) {
  await setItem(STORAGE_KEYS.GROUPS_LIST, {
    groups: groups,
    lastUpdated: Date.now()
  });
}

/**
 * Lấy danh sách groups
 * @returns {Promise<Object|null>}
 */
export async function getGroupsList() {
  return await getItem(STORAGE_KEYS.GROUPS_LIST);
}

/**
 * Lưu settings
 * @param {Object} settings
 */
export async function saveSettings(settings) {
  await setItem(STORAGE_KEYS.SETTINGS, settings);
}

/**
 * Lấy settings
 * @returns {Promise<Object>}
 */
export async function getSettings() {
  const settings = await getItem(STORAGE_KEYS.SETTINGS);
  // Return default settings if not found
  return settings || {
    delayMin: 15,
    delayMax: 90,
    retryCount: 3,
    anonymousPost: true
  };
}

/**
 * Thêm post vào history
 * @param {Object} postData
 */
export async function addToPostHistory(postData) {
  const history = await getItem(STORAGE_KEYS.POST_HISTORY) || [];
  history.unshift({
    ...postData,
    timestamp: Date.now()
  });
  // Giữ tối đa 100 entries
  if (history.length > 100) {
    history.pop();
  }
  await setItem(STORAGE_KEYS.POST_HISTORY, history);
}

/**
 * Lấy post history
 * @returns {Promise<Array>}
 */
export async function getPostHistory() {
  return await getItem(STORAGE_KEYS.POST_HISTORY) || [];
}

export { STORAGE_KEYS };
