/**
 * Core Auth Module - Cookie Management
 * Lấy cookies Facebook (c_user, xs, datr, sb, fr, etc.)
 */

// Danh sách cookies quan trọng cần lấy từ Facebook
const FB_COOKIE_NAMES = [
  'c_user',   // User ID
  'xs',       // Session token  
  'datr',     // Browser fingerprint
  'sb',       // Cookie security
  'fr',       // Facebook referrer
  'wd',       // Window dimensions
  'presence', // Online presence
  'locale',   // Language setting
];

/**
 * Lấy tất cả cookies từ domain Facebook
 * @returns {Promise<Object>} Object chứa cookies
 */
export async function getCookies() {
  return new Promise((resolve, reject) => {
    chrome.cookies.getAll({ domain: '.facebook.com' }, (cookies) => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
        return;
      }
      
      const cookieObj = {};
      cookies.forEach(cookie => {
        cookieObj[cookie.name] = cookie.value;
      });
      
      resolve(cookieObj);
    });
  });
}

/**
 * Lấy cookies được format thành chuỗi cho HTTP header
 * @returns {Promise<string>} Cookie string format: "name1=value1; name2=value2"
 */
export async function getCookieString() {
  const cookies = await getCookies();
  return Object.entries(cookies)
    .map(([name, value]) => `${name}=${value}`)
    .join('; ');
}

/**
 * Lấy các cookies quan trọng của Facebook
 * @returns {Promise<Object>} Object chứa essential cookies
 */
export async function getEssentialCookies() {
  const allCookies = await getCookies();
  const essential = {};
  
  FB_COOKIE_NAMES.forEach(name => {
    if (allCookies[name]) {
      essential[name] = allCookies[name];
    }
  });
  
  return essential;
}

/**
 * Kiểm tra xem user đã đăng nhập Facebook chưa
 * @returns {Promise<boolean>}
 */
export async function isLoggedIn() {
  const cookies = await getEssentialCookies();
  // Cần ít nhất c_user và xs để xác nhận đã login
  return !!(cookies.c_user && cookies.xs);
}

/**
 * Lấy User ID từ cookie c_user
 * @returns {Promise<string|null>}
 */
export async function getUserId() {
  const cookies = await getEssentialCookies();
  return cookies.c_user || null;
}

/**
 * Tạo object auth data với version để lưu vào storage
 * (Học từ Fewfeed pattern)
 * @returns {Promise<Object>}
 */
export async function createAuthData() {
  const cookies = await getEssentialCookies();
  const cookieString = await getCookieString();
  
  return {
    cookies: cookies,
    cookieString: cookieString,
    userId: cookies.c_user || null,
    version: 1, // Để tracking version nếu cần migrate
    timestamp: Date.now()
  };
}
