/**
 * Core API Module - fetchWithCookie và GraphQL helpers
 * Replay requests với cookies, retry logic, random delays
 */

import { getCookieString } from './auth.js';
import { randomDelay, delay, log, logError } from './utils.js';

const FB_GRAPHQL_ENDPOINT = 'https://www.facebook.com/api/graphql/';
const FB_UPLOAD_ENDPOINT = 'https://upload.facebook.com/ajax/react_composer/attachments/photo/upload';

/**
 * Fetch với Cookie header và retry logic
 * @param {string} url - URL to fetch
 * @param {Object} options - Fetch options
 * @param {Object} config - Config: { retries, delayMin, delayMax }
 * @returns {Promise<Response>}
 */
export async function fetchWithCookie(url, options = {}, config = {}) {
  const {
    retries = 3,
    delayMin = 1,
    delayMax = 3,
    useRandomDelay = false
  } = config;
  
  const cookieString = await getCookieString();
  
  const defaultHeaders = {
    'Cookie': cookieString,
    'User-Agent': navigator.userAgent,
    'Accept': '*/*',
    'Accept-Language': 'vi-VN,vi;q=0.9,en-US;q=0.8,en;q=0.7',
    'Cache-Control': 'no-cache',
    'Pragma': 'no-cache',
    'Sec-Fetch-Dest': 'empty',
    'Sec-Fetch-Mode': 'cors',
    'Sec-Fetch-Site': 'same-origin',
  };
  
  const mergedOptions = {
    ...options,
    credentials: 'include',
    headers: {
      ...defaultHeaders,
      ...options.headers
    }
  };
  
  let lastError = null;
  
  for (let attempt = 1; attempt <= retries; attempt++) {
    try {
      log(`Fetch attempt ${attempt}/${retries}: ${url.substring(0, 50)}...`);
      
      const response = await fetch(url, mergedOptions);
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }
      
      return response;
      
    } catch (error) {
      lastError = error;
      logError(`Attempt ${attempt} failed:`, error.message);
      
      if (attempt < retries) {
        if (useRandomDelay) {
          await randomDelay(delayMin, delayMax);
        } else {
          await delay(delayMin * 1000);
        }
      }
    }
  }
  
  throw lastError;
}

/**
 * Gửi GraphQL request tới Facebook
 * @param {string} docId - Document ID của query
 * @param {Object} variables - Query variables
 * @param {Object} extraParams - Extra form params
 * @returns {Promise<Object>} Response data
 */
export async function graphqlRequest(docId, variables, extraParams = {}) {
  const cookieString = await getCookieString();
  
  // Extract fb_dtsg từ page nếu cần
  // Trong background script, sẽ cần inject vào content script để lấy
  
  const formData = new FormData();
  formData.append('doc_id', docId);
  formData.append('variables', JSON.stringify(variables));
  formData.append('fb_api_caller_class', 'RelayModern');
  formData.append('server_timestamps', 'true');
  
  // Append extra params
  Object.entries(extraParams).forEach(([key, value]) => {
    formData.append(key, value);
  });
  
  const response = await fetchWithCookie(FB_GRAPHQL_ENDPOINT, {
    method: 'POST',
    body: formData
  }, {
    retries: 3,
    delayMin: 2,
    delayMax: 5
  });
  
  const text = await response.text();
  return parseGraphQLResponse(text);
}

/**
 * Parse Facebook GraphQL response
 * FB trả về format: "for (;;);{json}" hoặc multi-line JSON
 * @param {string} responseText
 * @returns {Object}
 */
export function parseGraphQLResponse(responseText) {
  try {
    // Remove "for (;;);" prefix nếu có
    let cleaned = responseText.replace(/^for \(;;\);/, '');
    
    // Thử parse directly
    try {
      return JSON.parse(cleaned);
    } catch {
      // Multi-line response, lấy line đầu tiên
      const lines = cleaned.split('\n').filter(line => line.trim());
      if (lines.length > 0) {
        return JSON.parse(lines[0]);
      }
    }
  } catch (error) {
    logError('Failed to parse GraphQL response:', error);
  }
  
  return null;
}

/**
 * Upload ảnh lên Facebook
 * @param {File} file - File object
 * @param {Object} params - Extra params (userId, etc.)
 * @returns {Promise<Object>} Upload result với photoID
 */
export async function uploadPhoto(file, params = {}) {
  const { userId } = params;
  
  const formData = new FormData();
  formData.append('source', '8');
  formData.append('profile_id', userId || '');
  formData.append('waterfallxapp', 'comet');
  formData.append('farr', file, file.name);
  formData.append('upload_id', `jsc_c_${Date.now()}`);
  
  const url = `${FB_UPLOAD_ENDPOINT}?av=${userId}&__aaid=0&__user=${userId}&__a=1`;
  
  const response = await fetchWithCookie(url, {
    method: 'POST',
    body: formData
  }, {
    retries: 3,
    delayMin: 1,
    delayMax: 3
  });
  
  const text = await response.text();
  const result = parseGraphQLResponse(text);
  
  return result?.payload?.photoID || null;
}

/**
 * Tạo request params chuẩn cho FB GraphQL
 * @param {Object} siteData - SiteData object từ page
 * @param {Object} dtsgData - fb_dtsg data
 * @returns {Object}
 */
export function createBaseParams(siteData = {}, dtsgData = {}) {
  return {
    __a: '1',
    __ccg: 'EXCELLENT',
    __req: Math.floor(Math.random() * 100).toString(),
    __hs: siteData.haste_session || '',
    __rev: siteData.server_revision || '',
    __hsi: siteData.hsi || '',
    __comet_req: siteData.comet_env || '',
    __spin_r: siteData.__spin_r || '',
    __spin_b: siteData.__spin_b || '',
    __spin_t: siteData.__spin_t || '',
    fb_dtsg: dtsgData.token || '',
    dpr: '1'
  };
}

export {
  FB_GRAPHQL_ENDPOINT,
  FB_UPLOAD_ENDPOINT
};
