/**
 * Service Worker v11.1 - FB Power Tools
 * Full Groups Features + Anonymous
 * 
 * Handlers:
 * - INIT_SET_COOKIE: Auto save từ content
 * - SAVE_GROUPS: Lưu groups từ inject
 * - SET_ANONYMOUS: Toggle anonymous mode
 * - POST_GROUPS: Post với anonymous support
 */
console.log('[SW] Service Worker v11.1 loaded at', new Date().toISOString());

import { isLoggedIn, createAuthData, getUserId } from '../core/auth.js';
import { saveAuthData, getAuthData, saveGroupsList, getGroupsList, getSettings, saveSettings } from '../core/storage.js';
import { bulkPost, bulkPostMulti } from '../modules/groups/poster.js';
import { syncToWebPlatform, resetSyncStatus } from '../core/sync.js';
import { ACTION_DOC_IDS } from '../modules/groups/actions.js';

const LOG = '[FB SW]';
const log = (...args) => console.log(LOG, ...args);
const logError = (...args) => console.error(LOG, ...args);

// ==================== State ====================

let pendingScan = null;
let cachedCookie = null;
let isAnonymousMode = false;

// ==================== Utilities ====================

async function findFacebookTab() {
  const tabs = await chrome.tabs.query({ url: '*://*.facebook.com/*' });
  return tabs.find(t => !t.url.includes('messenger.com')) || null;
}

function broadcast(msg) {
  chrome.runtime.sendMessage(msg).catch(() => {});
}

// ==================== Scan Logic ====================

async function scanGroups() {
  log('Starting scan...');
  
  const fbTab = await findFacebookTab();
  if (!fbTab) {
    return { success: false, error: 'Vui lòng mở facebook.com' };
  }
  
  log('Found FB tab:', fbTab.id);
  
  return new Promise(async (resolve) => {
    pendingScan = {
      resolve,
      startTime: Date.now()
    };
    
    // Send to content script
    try {
      await chrome.tabs.sendMessage(fbTab.id, { type: 'SCAN_GROUPS' });
      log('Sent SCAN_GROUPS to content');
    } catch (e) {
      log('Injecting content script...');
      await chrome.scripting.executeScript({
        target: { tabId: fbTab.id },
        files: ['content/content.js']
      }).catch(() => {});
      
      await new Promise(r => setTimeout(r, 500));
      
      try {
        await chrome.tabs.sendMessage(fbTab.id, { type: 'SCAN_GROUPS' });
        log('Sent after inject');
      } catch (e2) {
        pendingScan = null;
        resolve({ success: false, error: 'Cannot inject' });
        return;
      }
    }
    
    // Long timeout (60s for many pages)
    setTimeout(() => {
      if (pendingScan) {
        log('Scan timeout (60s) - checking storage...');
        // Don't fail - check if groups were saved
        getGroupsList().then(stored => {
          if (stored?.length > 0) {
            log('Found', stored.length, 'groups in storage');
            if (pendingScan) {
              pendingScan.resolve({ success: true, groups: stored, count: stored.length });
              pendingScan = null;
            }
          } else {
            if (pendingScan) {
              pendingScan.resolve({ success: false, error: 'Timeout - no groups received' });
              pendingScan = null;
            }
          }
        });
      }
    }, 60000);
  });
}

// ==================== Message Handlers ====================

const handlers = {
  // === Auth ===
  async GET_COOKIE() {
    const authData = await createAuthData();
    await saveAuthData(authData);
    return { success: true, data: authData };
  },
  
  async CHECK_LOGIN() {
    const loggedIn = await isLoggedIn();
    return { success: true, isLoggedIn: loggedIn, userId: loggedIn ? await getUserId() : null };
  },
  
  async GET_AUTH() {
    return { success: true, data: await getAuthData() };
  },
  
  // === Cookie from Content (Fewfeed Pattern) ===
  async INIT_SET_COOKIE(data) {
    cachedCookie = data;
    log('Cookie cached for:', data.userId);
    
    // Save to storage
    await chrome.storage.local.set({ fb_cookie: data });
    return { ok: true };
  },
  
  // === Groups ===
  async SCAN_GROUPS() {
    return await scanGroups();
  },
  
  async AUTO_SCAN_GROUPS() {
    const stored = await getGroupsList();
    if (stored?.groups?.length > 0) {
      return { success: true, data: stored.groups, count: stored.groups.length, cached: true };
    }
    return await scanGroups();
  },
  
  async GET_GROUPS() {
    const stored = await getGroupsList();
    return { success: true, data: stored?.groups || [] };
  },
  
  // === Real-time Progress from Inject ===
  async GROUPS_PROGRESS(data) {
    // Broadcast immediately so dashboard updates in real-time
    broadcast({ type: 'GROUPS_UPDATED', data: { 
      groups: data.groups, 
      inProgress: true,
      loadingRole: data.loadingRole,
      page: data.page
    } });
    return { ok: true };
  },
  
  // === Save Groups from Inject ===
  async SAVE_GROUPS(data) {
    const { success, groups, stats, error } = data;
    
    if (success && groups?.length > 0) {
      log(`Saved ${groups.length} groups (${stats?.admin || 0} Admin, ${stats?.member || 0} Member)`);
      
      await saveGroupsList(groups);
      
      // Save stats
      await chrome.storage.local.set({ fb_groups_stats: stats });
      
      // Broadcast immediately so dashboard updates
      broadcast({ type: 'GROUPS_UPDATED', data: { groups, stats } });
      
      // Resolve pending scan
      if (pendingScan) {
        pendingScan.resolve({ success: true, groups, count: groups.length, stats });
        pendingScan = null;
      }

      // Sync to web platform after groups saved
      try {
        const syncResult = await syncToWebPlatform();
        log('Web sync after save:', syncResult.success ? 'OK' : syncResult.error);
      } catch (e) {
        log('Web sync error:', e.message);
      }
    } else {
      log('Scan failed:', error);
      
      if (pendingScan) {
        pendingScan.resolve({ success: false, error: error || 'No groups' });
        pendingScan = null;
      }
    }
    
    return { ok: true };
  },
  
  // === Anonymous Mode ===
  async SET_ANONYMOUS(data) {
    isAnonymousMode = !!data.value;
    await chrome.storage.local.set({ fb_anonymous: isAnonymousMode });
    log('Anonymous mode:', isAnonymousMode);
    return { ok: true, value: isAnonymousMode };
  },
  
  async GET_ANONYMOUS() {
    const result = await chrome.storage.local.get('fb_anonymous');
    return { ok: true, value: !!result.fb_anonymous };
  },
  
  // === Page Data ===
  async GET_PAGE_DATA() {
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'No FB tab' };
    
    const results = await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      world: 'MAIN',
      func: () => {
        let userId, dtsg;
        if (typeof require === 'function') {
          try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
          try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
        }
        if (!userId) {
          const m = document.cookie.match(/c_user=(\d+)/);
          if (m) userId = m[1];
        }
        return { success: !!(userId && dtsg), userId, dtsg };
      }
    });
    return results?.[0]?.result || { success: false };
  },
  
  // === Post with Anonymous Support ===
  async POST_GROUPS(data) {
    const { groupIds, content, imageData, posts } = data;
    let { isAnonymous } = data;
    
    if (isAnonymous === undefined) {
      const stored = await chrome.storage.local.get('fb_anonymous');
      isAnonymous = !!stored.fb_anonymous;
    }
    
    // Multi-content mode: posts = [{groupId, message}]
    const isMultiContent = Array.isArray(posts) && posts.length > 0;
    
    if (!isMultiContent && !groupIds?.length) return { success: false, error: 'No groups' };
    if (!isMultiContent && !content?.trim()) return { success: false, error: 'No content' };
    
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    
    // Build the posting list: [{groupId, message}]
    const postList = isMultiContent 
      ? posts 
      : groupIds.map(gid => ({ groupId: gid, message: content }));
    
    const targetCount = postList.length;
    log(`Posting to ${targetCount} groups (multi: ${isMultiContent}, anonymous: ${isAnonymous})`);
    
    const settings = await getSettings();
    const minDelay = data.minDelay || settings.minDelay || 5000;
    const maxDelay = data.maxDelay || settings.maxDelay || 10000;
    
    const results = {
      total: targetCount,
      success: 0,
      failed: 0,
      posts: [],
      logs: []
    };
    
    // Post to each group one by one using executeScript on the Facebook tab
    for (let i = 0; i < postList.length; i++) {
      const { groupId, message: msg } = postList[i];
      
      broadcast({ type: 'POST_PROGRESS', data: { current: i + 1, total: targetCount, groupId, status: 'posting' } });
      
      try {
        const imageDataArray = data.imageDataArray || [];
        const textFormatPresetId = data.textFormatPresetId || '0';
        
        log(`[POST] Group ${groupId}: images=${imageDataArray.length}, anon=${isAnonymous}, bg=${textFormatPresetId}`);
        
        // ===== STEP 1: Discover anonymous actor ID FIRST (needed for photo upload) =====
        let anonymousActorId = null;
        if (isAnonymous) {
          log(`[ANON] Starting actor discovery for group ${groupId}...`);
          try {
            const anonResult = await chrome.scripting.executeScript({
              target: { tabId: tab.id },
              world: 'MAIN',
              args: [groupId],
              func: async (groupId) => {
                try {
                  let userId, dtsg, siteData;
                  if (typeof require === 'function') {
                    try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
                    try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
                    try { siteData = require('SiteData'); } catch(e) {}
                  }
                  if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
                  if (!userId || !dtsg) return { error: 'No auth for anon lookup' };
                  
                  const graphqlUrl = window._http_fb_api_graphql || 'https://www.facebook.com/api/graphql/';
                  const fd = new FormData();
                  fd.append('av', userId);
                  fd.append('__aaid', '0');
                  fd.append('__user', userId);
                  fd.append('__a', '1');
                  fd.append('__req', Math.floor(1000 * Math.random()).toString());
                  if (siteData) {
                    fd.append('__hs', siteData.haste_session || '');
                    fd.append('__ccg', 'EXCELLENT');
                    fd.append('__rev', siteData.server_revision || '');
                    fd.append('__hsi', siteData.hsi || '');
                    fd.append('__comet_req', siteData.comet_env || '15');
                    fd.append('__spin_r', siteData.__spin_r || '');
                    fd.append('__spin_b', siteData.__spin_b || '');
                    fd.append('__spin_t', siteData.__spin_t || '');
                  }
                  fd.append('dpr', '1');
                  fd.append('fb_dtsg', dtsg);
                  fd.append('fb_api_caller_class', 'RelayModern');
                  fd.append('fb_api_req_friendly_name', 'CometGroupDiscussionRootSuccessQuery');
                  
                  const variables = {
                    creative_provider_id: null, feedLocation: 'GROUP', feedbackSource: 0,
                    focusCommentID: null, feedType: 'DISCUSSION', scale: 1,
                    sortingSetting: 'CHRONOLOGICAL', filter_topic_id: null,
                    useDefaultActor: false, groupID: groupId,
                    hoistStories: [], hoistStoriesCount: 0,
                    hoistedSectionHeaderType: 'notifications', hasHoistStories: false,
                    privacySelectorRenderLocation: 'COMET_STREAM', renderLocation: 'group',
                    regular_stories_count: 1, regular_stories_stream_initial_count: 1,
                    regular_stories_cursor: null, shouldDeferMainFeed: false,
                    threadID: '', autoOpenChat: false, autoCreateChatDialog: false,
                    '__relay_internal__pv__CometUFIShareActionMigrationrelayprovider': true,
                    '__relay_internal__pv__GHLShouldChangeSponsoredDataFieldNamerelayprovider': true,
                    '__relay_internal__pv__GHLShouldChangeAdIdFieldNamerelayprovider': true,
                    '__relay_internal__pv__CometUFI_dedicated_comment_routable_dialog_gkrelayprovider': false,
                    '__relay_internal__pv__CometUFICommentAvatarStickerAnimatedImagerelayprovider': false,
                    '__relay_internal__pv__IsWorkUserrelayprovider': false,
                    '__relay_internal__pv__CometUFIReactionsEnableShortNamerelayprovider': false,
                    '__relay_internal__pv__TestPilotShouldIncludeDemoAdUseCaserelayprovider': false,
                    '__relay_internal__pv__FBReels_deprecate_short_form_video_context_gkrelayprovider': true,
                    '__relay_internal__pv__FeedDeepDiveTopicPillThreadViewEnabledrelayprovider': false,
                    '__relay_internal__pv__FBReels_enable_view_dubbed_audio_type_gkrelayprovider': true,
                    '__relay_internal__pv__CometImmersivePhotoCanUserDisable3DMotionrelayprovider': false,
                    '__relay_internal__pv__WorkCometIsEmployeeGKProviderrelayprovider': false,
                    '__relay_internal__pv__IsMergQAPollsrelayprovider': false,
                    '__relay_internal__pv__FBReels_enable_meta_ai_label_gkrelayprovider': true,
                    '__relay_internal__pv__FBReelsMediaFooter_comet_enable_reels_ads_gkrelayprovider': true,
                    '__relay_internal__pv__StoriesArmadilloReplyEnabledrelayprovider': true,
                    '__relay_internal__pv__FBReelsIFUTileContent_reelsIFUPlayOnHoverrelayprovider': true,
                    '__relay_internal__pv__GroupsCometGYSJFeedItemHeightrelayprovider': 206,
                    '__relay_internal__pv__StoriesShouldIncludeFbNotesrelayprovider': false,
                    '__relay_internal__pv__GHLShouldChangeSponsoredAuctionDistanceFieldNamerelayprovider': true,
                    '__relay_internal__pv__GHLShouldUseSponsoredAuctionLabelFieldNameV1relayprovider': true,
                    '__relay_internal__pv__GHLShouldUseSponsoredAuctionLabelFieldNameV2relayprovider': false,
                    '__relay_internal__pv__GroupsCometGroupChatLazyLoadLastMessageSnippetrelayprovider': false,
                    '__relay_internal__pv__GroupsCometLazyLoadFeaturedSectionrelayprovider': false
                  };
                  
                  fd.append('variables', JSON.stringify(variables));
                  fd.append('server_timestamps', 'true');
                  fd.append('doc_id', '25274469595580898');
                  
                  const resp = await fetch(graphqlUrl, {
                    method: 'POST', credentials: 'include', body: fd
                  });
                  
                  const text = await resp.text();
                  let cleaned = text.replace('for (;;);', '').trim();
                  const lines = cleaned.split('\n');
                  let actorId = null;
                  
                  for (const line of lines) {
                    const t = line.trim();
                    if (!t || !t.startsWith('{')) continue;
                    try {
                      const json = JSON.parse(t);
                      const edges = json?.data?.group?.comet_inline_composer_renderer?.group?.available_actors?.edges;
                      if (edges && edges.length > 0 && edges[0]?.node?.id) {
                        actorId = edges[0].node.id;
                        break;
                      }
                    } catch(e) {}
                  }
                  
                  if (!actorId) return { error: 'No anonymous actor found' };
                  return { actorId, userId };
                } catch(err) {
                  return { error: err.message };
                }
              }
            });
            
            const anonData = anonResult?.[0]?.result;
            if (anonData?.error) {
              logError(`Anonymous actor lookup error: ${anonData.error}`);
            } else if (anonData?.actorId) {
              anonymousActorId = anonData.actorId;
              log(`✓ Anonymous actor ID: ${anonymousActorId}`);
            } else {
              log(`⚠ No anonymous actor found, posting as self`);
            }
          } catch(e) {
            logError(`Anonymous actor lookup failed: ${e.message}`);
          }
        }
        
        // ===== STEP 2: Upload images (using anonymous actor ID if available) =====
        const effectiveUploadActorId = (isAnonymous && anonymousActorId) ? anonymousActorId : null;
        const uploadedPhotoIds = [];
        for (let idx = 0; idx < imageDataArray.length; idx++) {
          try {
            broadcast({ type: 'POST_PROGRESS', data: { current: i + 1, total: targetCount, groupId, status: `uploading_photo_${idx + 1}` } });
            
            const photoResult = await chrome.scripting.executeScript({
              target: { tabId: tab.id },
              world: 'MAIN',
              args: [imageDataArray[idx], effectiveUploadActorId],
              func: async (base64Data, actorIdOverride) => {
                try {
                  let userId, dtsg, lsd = '';
                  if (typeof require === 'function') {
                    try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
                    try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
                    try { lsd = require('LSD')?.token || ''; } catch(e) {}
                  }
                  if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
                  if (!userId || !dtsg) return { success: false, error: 'No auth for upload' };
                  
                  // Use anonymous actor ID for upload if provided
                  const uploadActorId = actorIdOverride || userId;
                  
                  const parts = base64Data.split(',');
                  const mime = parts[0].match(/:(.*?);/)?.[1] || 'image/jpeg';
                  const binary = atob(parts[1]);
                  const bytes = new Uint8Array(binary.length);
                  for (let j = 0; j < binary.length; j++) bytes[j] = binary.charCodeAt(j);
                  const blob = new Blob([bytes], { type: mime });
                  
                  const ext = mime.includes('png') ? '.png' : mime.includes('gif') ? '.gif' : '.jpg';
                  const fileName = `upload_${Date.now()}${ext}`;
                  
                  const formData = new FormData();
                  formData.append('source', '8');
                  formData.append('profile_id', uploadActorId);
                  formData.append('waterfallxapp', 'comet');
                  formData.append('farr', blob, fileName);
                  formData.append('upload_id', `jsc_c_${Date.now()}`);
                  formData.append('fb_dtsg', dtsg);
                  formData.append('jazoest', '25296');
                  if (lsd) formData.append('lsd', lsd);
                  
                  const url = `https://upload.facebook.com/ajax/react_composer/attachments/photo/upload?av=${uploadActorId}&__user=${userId}&__a=1`;
                  console.log('[FB-UPLOAD] Using actor:', uploadActorId, 'user:', userId);
                  const resp = await fetch(url, { method: 'POST', body: formData, credentials: 'include' });
                  const text = await resp.text();
                  let cleaned = text.trim();
                  if (cleaned.startsWith('for (;;);')) cleaned = cleaned.substring(9).trim();
                  const json = JSON.parse(cleaned);
                  const photoId = json?.payload?.photoID || json?.payload?.photo_id;
                  if (photoId) return { success: true, photoId };
                  return { success: false, error: 'No photoID: ' + cleaned.substring(0, 100) };
                } catch(err) {
                  return { success: false, error: err.message };
                }
              }
            });
            
            const uploadResult = photoResult?.[0]?.result;
            if (uploadResult?.success) {
              uploadedPhotoIds.push(uploadResult.photoId);
              log(`✓ Photo ${idx+1} uploaded: ${uploadResult.photoId}`);
            } else {
              logError(`✗ Photo ${idx+1} upload failed: ${uploadResult?.error || 'Script failed'}`);
            }
          } catch(uploadErr) {
            logError(`✗ Photo ${idx+1} upload error: ${uploadErr.message}`);
          }
        }
        
        log(`[POST] anonymousActorId=${anonymousActorId}, photos=${uploadedPhotoIds.length}`);

        // ===== STEP 2: Post with only the small photoId array + settings =====
        const scriptResults = await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          world: 'MAIN',
          args: [groupId, msg, isAnonymous ? '32972562495725502' : '25773390049008020', uploadedPhotoIds, isAnonymous, textFormatPresetId, anonymousActorId],
          func: async (groupId, message, DOC_ID, photoIds, isAnonymous, textFormatPresetId, anonymousActorId) => {
            let userId, dtsg;
            if (typeof require === 'function') {
              try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
              try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            }
            if (!userId) {
              const m = document.cookie.match(/c_user=(\d+)/);
              if (m) userId = m[1];
            }
            let lsd = '';
            if (typeof require === 'function') {
              try { lsd = require('LSD')?.token || ''; } catch(e) {}
            }
            if (!lsd) {
              const el = document.querySelector('input[name="lsd"]');
              if (el) lsd = el.value;
            }
            if (!userId || !dtsg) return { success: false, error: 'No auth', groupId };
            
            console.log('[FB-POST] Params:', { groupId, isAnonymous, textFormatPresetId, photoIds: photoIds?.length || 0 });
            
            const composerSessionId = crypto.randomUUID();
            const clientMutationId = Math.floor(Math.random() * 1000).toString();
            
            const inputObj = {
              composer_entry_point: isAnonymous ? "publisher_bar_anonymous_author" : "inline_composer",
              composer_source_surface: "group",
              composer_type: "group",
              source: "WWW",
              message: { ranges: [], text: message },
              with_tags_ids: null,
              inline_activities: [],
              text_format_preset_id: textFormatPresetId || "0",
              composed_text: {
                block_data: ["{}" ],
                block_depths: [0],
                block_types: [0],
                blocks: [message],
                entities: ["[]"],
                entity_map: "{}",
                inline_styles: ["[]"]
              },
              event_share_metadata: { surface: "newsfeed" },
              audience: { to_id: groupId },
              actor_id: (isAnonymous && anonymousActorId) ? anonymousActorId : userId,
              client_mutation_id: clientMutationId
            };
            
            // Anonymous posting fields (only added when anonymous)
            if (isAnonymous) {
              inputObj.group_flair = { flair_id: null };
              inputObj.ask_admin_to_post_for_user = { is_asking_admin_to_post: true };
              console.log('[FB-POST] Anonymous mode enabled, actor_id:', inputObj.actor_id);
            }
            
            // Attach uploaded photos
            if (photoIds && photoIds.length > 0) {
              inputObj.attachments = photoIds.map(pid => ({ photo: { id: pid } }));
              inputObj.text_format_preset_id = "0"; // no background with photos
              console.log('[FB-POST] Attaching photos:', photoIds);
            }
            
            const variables = {
              input: inputObj,
              feedLocation: "GROUP", feedbackSource: 0, focusCommentID: null,
              gridMediaWidth: null, groupID: null, scale: 1,
              privacySelectorRenderLocation: "COMET_STREAM",
              checkPhotosToReelsUpsellEligibility: false,
              renderLocation: "group", useDefaultActor: false,
              inviteShortLinkKey: null,
              isFeed: false, isFundraiser: false, isFunFactPost: false,
              isGroup: true, isEvent: false, isTimeline: false,
              isSocialLearning: false, isPageNewsFeed: false,
              isProfileReviews: false, isWorkSharedDraft: false,
              hashtag: null, canUserManageOffers: false,
              // Relay provider variables (matching working tool exactly)
              "__relay_internal__pv__CometUFIShareActionMigrationrelayprovider": true,
              "__relay_internal__pv__GHLShouldChangeSponsoredDataFieldNamerelayprovider": true,
              "__relay_internal__pv__GHLShouldChangeAdIdFieldNamerelayprovider": true,
              "__relay_internal__pv__CometUFI_dedicated_comment_routable_dialog_gkrelayprovider": false,
              "__relay_internal__pv__CometUFICommentAvatarStickerAnimatedImagerelayprovider": false,
              "__relay_internal__pv__IsWorkUserrelayprovider": false,
              "__relay_internal__pv__CometUFIReactionsEnableShortNamerelayprovider": false,
              "__relay_internal__pv__TestPilotShouldIncludeDemoAdUseCaserelayprovider": false,
              "__relay_internal__pv__FBReels_deprecate_short_form_video_context_gkrelayprovider": true,
              "__relay_internal__pv__FeedDeepDiveTopicPillThreadViewEnabledrelayprovider": false,
              "__relay_internal__pv__FBReels_enable_view_dubbed_audio_type_gkrelayprovider": true,
              "__relay_internal__pv__CometImmersivePhotoCanUserDisable3DMotionrelayprovider": false,
              "__relay_internal__pv__WorkCometIsEmployeeGKProviderrelayprovider": false,
              "__relay_internal__pv__IsMergQAPollsrelayprovider": false,
              "__relay_internal__pv__FBReels_enable_meta_ai_label_gkrelayprovider": true,
              "__relay_internal__pv__FBReelsMediaFooter_comet_enable_reels_ads_gkrelayprovider": true,
              "__relay_internal__pv__StoriesArmadilloReplyEnabledrelayprovider": true,
              "__relay_internal__pv__FBReelsIFUTileContent_reelsIFUPlayOnHoverrelayprovider": true,
              "__relay_internal__pv__GroupsCometGYSJFeedItemHeightrelayprovider": 206,
              "__relay_internal__pv__StoriesShouldIncludeFbNotesrelayprovider": false,
              "__relay_internal__pv__GHLShouldChangeSponsoredAuctionDistanceFieldNamerelayprovider": true,
              "__relay_internal__pv__GHLShouldUseSponsoredAuctionLabelFieldNameV1relayprovider": true,
              "__relay_internal__pv__GHLShouldUseSponsoredAuctionLabelFieldNameV2relayprovider": false
            };
            
            // Get SiteData for full params (matching working tool)
            let siteData = null;
            if (typeof require === 'function') {
              try { siteData = require('SiteData'); } catch(e) {}
            }
            
            const bodyParams = new FormData();
            const effectiveActorId = (isAnonymous && anonymousActorId) ? anonymousActorId : userId;
            bodyParams.append('av', effectiveActorId);
            bodyParams.append('__aaid', '0');
            bodyParams.append('__user', userId);
            bodyParams.append('__a', '1');
            bodyParams.append('__req', Math.floor(1000 * Math.random()).toString());
            if (siteData) {
              bodyParams.append('__hs', siteData.haste_session || '');
              bodyParams.append('__ccg', 'EXCELLENT');
              bodyParams.append('__rev', siteData.server_revision || '');
              bodyParams.append('__hsi', siteData.hsi || '');
              bodyParams.append('__comet_req', siteData.comet_env || '15');
              bodyParams.append('__spin_r', siteData.__spin_r || '');
              bodyParams.append('__spin_b', siteData.__spin_b || '');
              bodyParams.append('__spin_t', siteData.__spin_t || '');
            } else {
              bodyParams.append('__comet_req', '15');
            }
            bodyParams.append('dpr', '1');
            bodyParams.append('fb_dtsg', dtsg);
            bodyParams.append('jazoest', '25296');
            if (lsd) bodyParams.append('lsd', lsd);
            bodyParams.append('__crn', 'comet.fbweb.CometGroupDiscussionRoute');
            bodyParams.append('fb_api_caller_class', 'RelayModern');
            bodyParams.append('fb_api_req_friendly_name', 'ComposerStoryCreateMutation');
            bodyParams.append('server_timestamps', 'true');
            bodyParams.append('doc_id', DOC_ID);
            bodyParams.append('variables', JSON.stringify(variables));
            
            try {
              const graphqlUrl = window._http_fb_api_graphql || 'https://www.facebook.com/api/graphql/';
              const response = await fetch(graphqlUrl, {
                method: 'POST', credentials: 'include',
                body: bodyParams
              });
              
              const text = await response.text();
              let cleaned = text.trim();
              if (cleaned.startsWith('for (;;);')) cleaned = cleaned.substring(9).trim();
              
              let postId = null;
              let fbError = null;
              
              const tryExtractPost = (json) => {
                if (json?.data?.story_create) {
                  const sc = json.data.story_create;
                  return sc.story?.legacy_story_hideable_id || sc.post_id || sc.story?.id || null;
                }
                // Anonymous posts may return a different structure (pending approval)
                if (json?.data?.group_post_submit_for_approval) {
                  return 'pending_approval';
                }
                return null;
              };
              
              try {
                const json = JSON.parse(cleaned);
                postId = tryExtractPost(json);
                if (!postId && json.error) fbError = `FB error ${json.error}: ${json.errorSummary || json.errorDescription || 'Unknown'}`;
                if (!postId && json.errors) fbError = `FB errors: ${JSON.stringify(json.errors).substring(0, 200)}`;
              } catch(e) {
                const lines = cleaned.split('\n');
                for (const line of lines) {
                  const trimmed = line.trim();
                  if (!trimmed || !trimmed.startsWith('{')) continue;
                  try {
                    const json = JSON.parse(trimmed);
                    const id = tryExtractPost(json);
                    if (id) { postId = id; break; }
                    if (json.error && !fbError) fbError = `FB error ${json.error}: ${json.errorSummary || ''}`;
                    if (json.errors && !fbError) fbError = `FB errors: ${JSON.stringify(json.errors).substring(0, 200)}`;
                  } catch(e2) { /* skip */ }
                }
              }
              
              console.log('[FB-POST] Result:', { postId, fbError, responseSnippet: cleaned.substring(0, 500) });
              
              if (postId) return { success: true, postId, groupId, photosUploaded: photoIds?.length || 0 };
              if (fbError) return { success: false, error: fbError, groupId };
              return { success: false, error: `No postId in ${text.length}B: ${cleaned.substring(0, 300)}`, groupId };
            } catch(err) {
              return { success: false, error: err.message, groupId };
            }
          }
        });
        
        const result = scriptResults?.[0]?.result || { success: false, error: 'Script failed', groupId };
        results.posts.push(result);
        
        if (result.success) {
          results.success++;
          results.logs.push(`✓ ${groupId}: ${result.postId}`);
          log(`✓ Posted to ${groupId}: ${result.postId}`);
        } else {
          results.failed++;
          results.logs.push(`✗ ${groupId}: ${result.error}`);
          logError(`✗ Failed ${groupId}: ${result.error}`);
        }
        
      } catch (err) {
        results.posts.push({ success: false, error: err.message, groupId });
        results.failed++;
        results.logs.push(`✗ ${groupId}: ${err.message}`);
        logError(`✗ Exception ${groupId}: ${err.message}`);
      }
      
      broadcast({ type: 'POST_PROGRESS', data: { 
        current: i + 1, total: targetCount, groupId, 
        status: results.posts[i].success ? 'success' : 'failed', 
        result: results.posts[i] 
      }});
      
      // Delay between posts
      if (i < postList.length - 1) {
        const waitTime = Math.floor(Math.random() * (maxDelay - minDelay + 1) + minDelay);
        log(`Waiting ${waitTime / 1000}s...`);
        broadcast({ type: 'POST_PROGRESS', data: { current: i + 1, total: targetCount, status: 'waiting', waitTime: waitTime / 1000 }});
        await new Promise(r => setTimeout(r, waitTime));
      }
    }
    
    log(`Done: ${results.success}/${results.total} success`);
    const finalResults = { success: true, summary: results, results: results.posts };
    
    broadcast({ type: 'POST_COMPLETE', data: finalResults });
    
    // Save post log
    try {
      const logEntry = {
        id: `log_${Date.now()}`,
        timestamp: Date.now(),
        content: isMultiContent ? posts[0]?.message || '' : content,
        mode: isMultiContent ? 'multi' : 'uniform',
        isAnonymous,
        groups: results.posts.map(p => ({
          groupId: p.groupId,
          success: p.success,
          postId: p.postId || null,
          error: p.error || null,
          message: isMultiContent ? posts.find(x => x.groupId === p.groupId)?.message : content
        })),
        summary: {
          total: results.total,
          success: results.success,
          failed: results.failed
        },
        logs: results.logs
      };
      const stored = await chrome.storage.local.get('post_logs');
      const logs = stored.post_logs || [];
      logs.unshift(logEntry);
      if (logs.length > 100) logs.length = 100;
      await chrome.storage.local.set({ post_logs: logs });
      log('Post log saved:', logEntry.id);
    } catch (e) {
      logError('Failed to save post log:', e);
    }
    
    return { success: true, results: finalResults };
  },

  // === Post Logs ===
  async GET_POST_LOGS() {
    const stored = await chrome.storage.local.get('post_logs');
    return { success: true, logs: stored.post_logs || [] };
  },

  async CLEAR_POST_LOGS() {
    await chrome.storage.local.remove('post_logs');
    return { success: true };
  },

  // === Scheduling ===
  async SCHEDULE_POST(data) {
    const { scheduledTime, groupIds, content, posts, imageData, isAnonymous } = data;
    if (!scheduledTime) return { success: false, error: 'No scheduledTime' };
    
    const id = `sched_${Date.now()}`;
    const entry = {
      id, scheduledTime, groupIds, content, posts, imageData, isAnonymous,
      status: 'pending', createdAt: Date.now()
    };
    
    // Save to storage
    const stored = await chrome.storage.local.get('fb_scheduled_posts');
    const list = stored.fb_scheduled_posts || [];
    list.push(entry);
    await chrome.storage.local.set({ fb_scheduled_posts: list });
    
    // Set alarm
    const delayMs = scheduledTime - Date.now();
    if (delayMs > 0) {
      chrome.alarms.create(id, { when: scheduledTime });
      log(`Scheduled post ${id} for ${new Date(scheduledTime).toLocaleString()}`);
    } else {
      log(`Schedule time is in the past, posting immediately`);
      await executeScheduledPost(id);
    }
    
    return { success: true, id, scheduledTime };
  },

  async GET_SCHEDULED_POSTS() {
    const stored = await chrome.storage.local.get('fb_scheduled_posts');
    return { success: true, data: stored.fb_scheduled_posts || [] };
  },

  async DELETE_SCHEDULED_POST(data) {
    const { id } = data;
    if (!id) return { success: false, error: 'No id' };
    
    const stored = await chrome.storage.local.get('fb_scheduled_posts');
    const list = (stored.fb_scheduled_posts || []).filter(p => p.id !== id);
    await chrome.storage.local.set({ fb_scheduled_posts: list });
    chrome.alarms.clear(id);
    log(`Deleted scheduled post: ${id}`);
    return { success: true };
  },
  
  // === Settings ===
  async GET_SETTINGS() {
    return { success: true, data: await getSettings() };
  },
  
  async SAVE_SETTINGS(data) {
    await saveSettings(data);
    return { success: true };
  },

  // === Web Platform Sync ===
  async SYNC_WEB(data) {
    const forceGroups = data?.forceGroups || false;
    return await syncToWebPlatform(forceGroups);
  },

  async RESET_SYNC() {
    await resetSyncStatus();
    return { success: true };
  },

  // === Sync Status (for popup) ===
  async GET_SYNC_STATUS() {
    const syncData = await chrome.storage.local.get('fb_web_synced');
    return { 
      success: true, 
      lastSync: syncData.fb_web_synced?.lastSync || null 
    };
  },

  // === Profile Info (for Dashboard) ===
  async GET_PROFILE() {
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        world: 'MAIN',
        func: async () => {
          // Get required data
          let userId, dtsg, lsd;
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token; } catch(e) {}
          }
          if (!userId) {
            const m = document.cookie.match(/c_user=(\d+)/);
            if (m) userId = m[1];
          }
          if (!dtsg || !userId) return { success: false, error: 'Not logged in' };
          
          // GraphQL request for profile
          const params = new URLSearchParams({
            av: userId,
            __user: userId,
            __a: '1',
            fb_dtsg: dtsg,
            lsd: lsd || '',
            fb_api_caller_class: 'RelayModern',
            fb_api_req_friendly_name: 'ProfileCometHeaderQuery',
            variables: JSON.stringify({
              scale: 1,
              selectedID: userId,
              selectedSpaceType: 'profile',
              shouldUseFXIMProfilePicEditor: false,
              userID: userId
            }),
            doc_id: '25964356583202181'
          });
          
          const resp = await fetch('https://www.facebook.com/api/graphql/', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: params.toString(),
            credentials: 'include'
          });
          
          const text = await resp.text();
          // Parse first JSON line (FB returns multiple JSON objects)
          const firstLine = text.split('\n')[0];
          const data = JSON.parse(firstLine);
          
          const user = data?.data?.user?.profile_header_renderer?.user;
          if (!user) {
            console.warn('Cannot parse full profile, returning basic ID');
            return {
              success: true, // Mark as partial success
              partial: true,
              profile: {
                id: userId,
                name: 'Facebook User',
                avatar: `https://graph.facebook.com/${userId}/picture?type=large`
              }
            };
          }
          
          return {
            success: true,
            profile: {
              id: userId,
              name: user.name,
              avatar: user.profilePicLarge?.uri || user.profilePicMedium?.uri,
              cover: user.cover_photo?.photo?.image?.uri,
              url: user.url
            }
          };
        }
      });
      
      const result = results?.[0]?.result;
      if (!result?.success && result?.error === 'Not logged in') {
        return result;
      }
      
      // If script failed entirely, try to return basic stored ID
      if (!result || !result.success) {
        const storedUserId = await getUserId();
        if (storedUserId) {
          return {
            success: true,
            partial: true,
            profile: {
              id: storedUserId,
              name: 'Facebook User',
              avatar: `https://graph.facebook.com/${storedUserId}/picture?type=large`
            }
          };
        }
      }

      return result || { success: false, error: 'Script failed' };
    } catch (e) {
      log('GET_PROFILE error:', e.message);
      return { success: false, error: e.message };
    }
  },

  // === Group Info (for Dashboard - single group) ===
  async GET_GROUP_INFO(data) {
    const { groupId, useCache = true } = data;
    if (!groupId) return { success: false, error: 'No groupId' };
    
    // Check cache first
    if (useCache) {
      const cached = await chrome.storage.local.get(`fb_group_info_${groupId}`);
      const info = cached[`fb_group_info_${groupId}`];
      if (info && info.timestamp > Date.now() - 24 * 60 * 60 * 1000) { // 24h cache
        return { success: true, data: info.data, cached: true };
      }
    }
    
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        world: 'MAIN',
        args: [groupId],
        func: async (gid) => {
          let userId, dtsg, lsd;
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token; } catch(e) {}
          }
          if (!userId) {
            const m = document.cookie.match(/c_user=(\d+)/);
            if (m) userId = m[1];
          }
          if (!dtsg || !userId) return { success: false, error: 'Not logged in' };
          
          const params = new URLSearchParams({
            av: userId,
            __user: userId,
            fb_dtsg: dtsg,
            lsd: lsd || '',
            fb_api_caller_class: 'RelayModern',
            fb_api_req_friendly_name: 'GroupsCometRootQuery',
            doc_id: '25810880338576777',
            __a: '1',
            variables: JSON.stringify({
              groupID: gid,
              inviteShortLinkKey: null,
              isChainingRecommendationUnit: false,
              scale: 1,
              __relay_internal__pv__GroupsCometGroupChatLazyLoadLastMessageSnippetrelayprovider: false
            })
          });

          try {
            const resp = await fetch('https://www.facebook.com/api/graphql/', {
              method: 'POST',
              headers: { 
                'Content-Type': 'application/x-www-form-urlencoded',
                'x-asbd-id': '359341',
                'x-fb-friendly-name': 'CometGroupRootQuery',
                'x-fb-lsd': lsd || ''
              },
              body: params.toString(),
              credentials: 'include'
            });
            
            const text = await resp.text();
            // Robust cleaning for Facebook's JSON prefix
            const cleanJsonText = text.replace(/^for \(;;|for \(;;\);|for\(;;\);/, '').trim();
            let json;
            try {
              json = JSON.parse(cleanJsonText);
            } catch (e) {
              return { success: false, error: 'JSON parse failed', debug: text.substring(0, 500) };
            }
            
            // Comprehensive group data extraction
            // Facebook can return multiple layers. We look for the one that actually has a name.
            const candidates = [
              json?.data?.group,
              json?.data?.node,
              json?.data?.profile_header_renderer?.group,
              json?.data?.group?.profile_header_renderer?.group
            ].filter(obj => obj && (obj.name || obj.featurable_title));

            const group = candidates[0] || json?.data?.group || json?.data?.node;

            if (!group) return { success: false, error: 'Cannot find group in response', debug: Object.keys(json?.data || {}) };
            
            // Member count extraction with multiple paths for redundancy
            let memberCount = 
              group.group_member_profiles?.facepile_profiles?.count || 
              group.if_viewer_can_see_members_facepile_in_entity_header?.facepile_profiles?.count || 
              group.group_member_profiles?.count || 
              group.viewer_post_status?.member_count || 
              group.group_members?.count ||
              null;
            
            const formattedText = group.group_member_profiles?.formatted_count_text || 
                                 group.group_member_profiles?.formatted_text || '';
            
            // Regex Fallback (e.g. "455.9K members")
            if (!memberCount && formattedText) {
              const match = formattedText.match(/([\d.,]+)([MK]?)/i);
              if (match) {
                let num = parseFloat(match[1].replace(/,/g, ''));
                const suffix = match[2].toUpperCase();
                if (suffix === 'K') num *= 1000;
                if (suffix === 'M') num *= 1000000;
                memberCount = Math.floor(num);
              }
            }

            // LAST RESORT: Recursive search for anything that looks like a member count
            if (!memberCount) {
              const deepSearch = (obj) => {
                if (!obj || typeof obj !== 'object') return null;
                for (const key in obj) {
                  if (key === 'count' && typeof obj[key] === 'number' && obj[key] > 100) return obj[key];
                  const res = deepSearch(obj[key]);
                  if (res) return res;
                }
                return null;
              };
              memberCount = deepSearch(json?.data);
            }

            return {
              success: true,
              data: {
                id: gid,
                name: group.featurable_title?.text || group.name || 'Unknown',
                memberCount: memberCount || 0,
                memberCountText: formattedText,
                privacy: group.privacy_info?.title?.text || group.privacy || '',
                cover: group.cover_renderer?.cover_photo_content?.photo?.image?.uri || 
                       group.cover_photo?.uri || 
                       group.profile_picture_for_sticky_bar?.uri,
                avatar: group.profile_picture_for_sticky_bar?.uri || group.profile_picture?.uri,
                url: group.url || `/groups/${gid}/`
              }
            };
          } catch (err) {
            return { success: false, error: err.message };
          }
        }
      });
      
      const result = results?.[0]?.result;
      if (result?.success) {
        // Cache the result
        await chrome.storage.local.set({
          [`fb_group_info_${groupId}`]: {
            data: result.data,
            timestamp: Date.now()
          }
        });
      }
      
      return result || { success: false, error: 'Script failed' };
    } catch (e) {
      log('GET_GROUP_INFO error:', e.message);
      return { success: false, error: e.message };
    }
  },

  // === Batch Group Info (with rate limiting) ===
  async BATCH_GROUP_INFO(data) {
    const { groupIds = [], delayMs = 3000, useCache = true } = data;
    if (!groupIds.length) return { success: false, error: 'No groupIds' };
    
    const results = {};
    const toFetch = [];
    
    // Check cache first for all groups
    for (const gid of groupIds) {
      if (useCache) {
        const cached = await chrome.storage.local.get(`fb_group_info_${gid}`);
        const info = cached[`fb_group_info_${gid}`];
        if (info && info.timestamp > Date.now() - 24 * 60 * 60 * 1000) { // 24h cache
          results[gid] = { ...info.data, cached: true };
          continue;
        }
      }
      toFetch.push(gid);
    }
    
    log(`BATCH_GROUP_INFO: ${groupIds.length} requested, ${Object.keys(results).length} cached, ${toFetch.length} to fetch`);
    
    // Fetch remaining groups with delay
    for (let i = 0; i < toFetch.length; i++) {
      const gid = toFetch[i];
      
      try {
        const result = await handlers.GET_GROUP_INFO({ groupId: gid, useCache: false });
        if (result.success) {
          results[gid] = result.data;
        } else {
          results[gid] = { id: gid, error: result.error };
        }
        
        // Broadcast progress AND data for real-time UI updates
        broadcast({ 
          type: 'GROUP_INFO_PROGRESS', 
          data: { 
            current: i + 1, 
            total: toFetch.length, 
            groupId: gid,
            success: result.success,
            details: result.success ? result.data : null 
          } 
        });

      } catch (e) {
        results[gid] = { id: gid, error: e.message };
      }
      
      // Delay between requests (except for the last one)
      if (i < toFetch.length - 1) {
        await new Promise(r => setTimeout(r, delayMs));
      }
    }
    
    return { success: true, data: results, fetched: toFetch.length, cached: Object.keys(results).length - toFetch.length };
  },

  // === Clear Group Info Cache ===
  async CLEAR_GROUP_INFO_CACHE() {
    const all = await chrome.storage.local.get(null);
    const keysToRemove = Object.keys(all).filter(k => k.startsWith('fb_group_info_'));
    if (keysToRemove.length > 0) {
      await chrome.storage.local.remove(keysToRemove);
    }
    log(`Cleared ${keysToRemove.length} group info cache entries`);
    return { success: true, cleared: keysToRemove.length };
  },

  // === Fetch Posts from Source Group (for Copy Posts feature) ===
  async FETCH_GROUP_POSTS(data) {
    const { sourceGroupId, count = 10, skipImageFetch = false, cursor: initialCursor = null } = data;
    if (!sourceGroupId) return { success: false, error: 'No sourceGroupId' };
    
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    
    log(`[FETCH_POSTS] Fetching ${count} posts from group ${sourceGroupId}`);
    
    try {
      const result = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        world: 'MAIN',
        args: [sourceGroupId, count, skipImageFetch, initialCursor],
        func: async (groupId, maxPosts, _skipImageFetch, _initialCursor) => {
          try {
            // Get auth data
            let userId, dtsg, lsd = '';
            if (typeof require === 'function') {
              try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
              try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
              try { lsd = require('LSD')?.token || ''; } catch(e) {}
            }
            if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
            if (!userId || !dtsg) return { success: false, error: 'No auth' };
            
            const graphqlUrl = window._http_fb_api_graphql || 'https://www.facebook.com/api/graphql/';
            let siteData = null;
            if (typeof require === 'function') {
              try { siteData = require('SiteData'); } catch(e) {}
            }
            
            // Helper: create FormData with common params
            function makeFormData() {
              const fd = new FormData();
              fd.append('av', userId);
              fd.append('__aaid', '0');
              fd.append('__user', userId);
              fd.append('__a', '1');
              fd.append('__req', String(Math.floor(1000 * Math.random())));
              if (siteData) {
                fd.append('__hs', siteData.haste_session || '');
                fd.append('__ccg', 'EXCELLENT');
                fd.append('__rev', siteData.server_revision || '');
                fd.append('__hsi', siteData.hsi || '');
                fd.append('__comet_req', siteData.comet_env || '15');
                fd.append('__spin_r', siteData.__spin_r || '');
                fd.append('__spin_b', siteData.__spin_b || '');
                fd.append('__spin_t', siteData.__spin_t || '');
              } else {
                fd.append('__comet_req', '15');
              }
              fd.append('dpr', '1');
              fd.append('fb_dtsg', dtsg);
              fd.append('jazoest', '25296');
              if (lsd) fd.append('lsd', lsd);
              fd.append('__crn', 'comet.fbweb.CometGroupDiscussionRoute');
              fd.append('fb_api_caller_class', 'RelayModern');
              fd.append('server_timestamps', 'true');
              return fd;
            }
            
            // Common relay variables
            const relayVars = `"__relay_internal__pv__GHLShouldChangeAdIdFieldNamerelayprovider":true,"__relay_internal__pv__GHLShouldChangeSponsoredDataFieldNamerelayprovider":true,"__relay_internal__pv__CometUFICommentAvatarStickerAnimatedImagerelayprovider":false,"__relay_internal__pv__IsWorkUserrelayprovider":false,"__relay_internal__pv__TestPilotShouldIncludeDemoAdUseCaserelayprovider":false,"__relay_internal__pv__FBReels_deprecate_short_form_video_context_gkrelayprovider":true,"__relay_internal__pv__FeedDeepDiveTopicPillThreadViewEnabledrelayprovider":false,"__relay_internal__pv__FBReels_enable_view_dubbed_audio_type_gkrelayprovider":true,"__relay_internal__pv__CometImmersivePhotoCanUserDisable3DMotionrelayprovider":false,"__relay_internal__pv__WorkCometIsEmployeeGKProviderrelayprovider":false,"__relay_internal__pv__IsMergQAPollsrelayprovider":false,"__relay_internal__pv__FBReels_enable_meta_ai_label_gkrelayprovider":true,"__relay_internal__pv__FBReelsMediaFooter_comet_enable_reels_ads_gkrelayprovider":true,"__relay_internal__pv__CometUFIReactionsEnableShortNamerelayprovider":false,"__relay_internal__pv__CometUFIShareActionMigrationrelayprovider":true,"__relay_internal__pv__CometUFI_dedicated_comment_routable_dialog_gkrelayprovider":false,"__relay_internal__pv__StoriesArmadilloReplyEnabledrelayprovider":true,"__relay_internal__pv__FBReelsIFUTileContent_reelsIFUPlayOnHoverrelayprovider":true,"__relay_internal__pv__GroupsCometGYSJFeedItemHeightrelayprovider":206,"__relay_internal__pv__StoriesShouldIncludeFbNotesrelayprovider":false`;
            
            // Helper: fetch from graphql
            async function gqlFetch(docId, friendlyName, variables) {
              const fd = makeFormData();
              fd.append('fb_api_req_friendly_name', friendlyName);
              fd.append('variables', variables);
              fd.append('doc_id', docId);
              const resp = await fetch(graphqlUrl, { method: 'POST', credentials: 'include', body: fd });
              return await resp.text();
            }
            
            // Helper: parse FB multi-line JSON response
            function parseMultiLineJson(text) {
              const cleaned = text.replace('for (;;);', '');
              const lines = cleaned.split('\n');
              const results = [];
              for (const line of lines) {
                if (!line.trim()) continue;
                try { results.push(JSON.parse(line)); } catch(e) {}
              }
              return results;
            }
            
            // Helper: safely get nested property
            function getVal(obj, path) {
              const parts = path.split('.');
              let cur = obj;
              for (const p of parts) {
                if (cur == null) return undefined;
                // Handle array index notation like [0]
                const arrMatch = p.match(/^(\w+)\[(\d+)\]$/);
                if (arrMatch) {
                  cur = cur[arrMatch[1]];
                  if (cur == null) return undefined;
                  cur = cur[parseInt(arrMatch[2])];
                } else {
                  cur = cur[p];
                }
              }
              return cur;
            }
            
            // Helper: extract media URLs from a post's attachments
            function extractMediaUrls(postData) {
              const urls = [];
              const attachments = postData.attachments || [];
              
              for (const att of attachments) {
                const styles = att?.styles?.attachment || att;
                
                // Try all known subattachment structures
                const subSources = [
                  styles.all_subattachments?.nodes,
                  styles.four_photos_subattachments?.nodes,
                  styles.five_photos_subattachments?.nodes,
                ];
                
                for (const nodes of subSources) {
                  if (nodes && nodes.length > 0) {
                    for (const node of nodes) {
                      const uri = node?.media?.viewer_image?.uri
                        || node?.media?.image?.uri
                        || node?.media?.large_share_image?.uri;
                      if (uri && !urls.includes(uri)) urls.push(uri);
                    }
                    if (urls.length > 0) break;
                  }
                }
                
                // Single image
                if (urls.length === 0) {
                  const singleUri = styles.media?.viewer_image?.uri
                    || styles.media?.photo_image?.uri
                    || styles.media?.image?.uri
                    || styles.media?.large_share_image?.uri;
                  if (singleUri && !urls.includes(singleUri)) urls.push(singleUri);
                }
              }
              
              return urls;
            }
            
            // ===== STEP 1: Fetch posts via pagination =====
            const allPosts = [];
            const seenIds = new Set();
            let cursor = _initialCursor || null;
            const perPage = 50; // Always request FB max to compensate for filtered non-Story items
            // Real-world: ~3-5 Story items per 50-item page (rest are GroupQuestion, ads, etc.)
            // No maxPages cap — keep fetching until we have enough or no more cursor
            let pageNum = 0;
            
            // Helper: extract post data from a node
            function extractPost(node) {
              if (!node) return null;
              const postId = node.post_id || getVal(node, 'feedback.owning_profile.id') || node.id;
              const storyId = node.id;
              if (!postId && !storyId) return null;
              const uniqueKey = postId || storyId;
              if (seenIds.has(uniqueKey)) return null;
              seenIds.add(uniqueKey);
              
              const authorName = getVal(node, 'comet_sections.context_layout.story.comet_sections.actor_photo.story.actors[0].name')
                || getVal(node, 'feedback.owning_profile.name')
                || 'Unknown';
              const message = getVal(node, 'comet_sections.content.story.message.text')
                || getVal(node, 'comet_sections.content.story.comet_sections.message.story.message.text')
                || '';
              const textFormatPresetId = getVal(node, 'comet_sections.content.story.text_format_metadata.preset_id') || '0';
              
              // Timestamp: try direct, then from metadata array
              let timestamp = getVal(node, 'comet_sections.timestamp.story.creation_time');
              if (!timestamp) {
                const meta = getVal(node, 'comet_sections.context_layout.story.comet_sections.metadata');
                if (Array.isArray(meta)) {
                  const tsMeta = meta.find(m => m?.__typename === 'CometFeedStoryMinimizedTimestampStrategy');
                  if (tsMeta) timestamp = getVal(tsMeta, 'story.creation_time');
                }
              }
              
              const mediaUrls = extractMediaUrls(node);
              const hasVideo = node.attachments?.some(att => {
                const mediaType = getVal(att, 'styles.attachment.media.__typename');
                return mediaType === 'Video' || mediaType === 'LiveVideo';
              }) || false;
              
              // Detect reshare: attached_story in content
              const attachedStory = getVal(node, 'comet_sections.content.story.attached_story')
                || getVal(node, 'attached_story');
              const isReshare = !!(attachedStory && (attachedStory.actors || attachedStory.comet_sections || attachedStory.message || attachedStory.url));
              
              return {
                post_id: postId || node.id,
                story_id: storyId,
                author_name: authorName,
                message,
                text_format_preset_id: textFormatPresetId,
                media_urls: mediaUrls,
                has_video: hasVideo,
                is_reshare: isReshare,
                timestamp
              };
            }
            
            while (allPosts.length < maxPosts) {
              pageNum++;
              const prevCount = allPosts.length;
              
              const vars = `{"count":${perPage},"cursor":${cursor ? JSON.stringify(cursor) : 'null'},"feedLocation":"GROUP","feedType":"DISCUSSION","feedbackSource":0,"filterTopicId":null,"focusCommentID":null,"privacySelectorRenderLocation":"COMET_STREAM","renderLocation":"group","scale":1,"sortingSetting":"CHRONOLOGICAL","stream_initial_count":1,"useDefaultActor":false,"id":"${groupId}",${relayVars}}`;
              
              console.log(`[FB SW] [FETCH_POSTS] Page ${pageNum}, have ${allPosts.length}/${maxPosts} posts so far`);
              let rawText = await gqlFetch(
                '24451884974509524', 
                'GroupsCometFeedRegularStoriesPaginationQuery', 
                vars
              );
              
              const cleaned = rawText.replace('for (;;);', '').trim();
              let foundPosts = false;
              let newCursor = null;
              
              // Parse all lines (works for both single JSON and multi-line streaming)
              const parsed = parseMultiLineJson(cleaned);
              
              // === Extract posts from ALL parsed entries ===
              for (const p of parsed) {
                if (allPosts.length >= maxPosts) break;
                
                // Path 1: Streaming label entries (Lines 1-N with label containing 'stream' + 'group_feed')
                if (p?.label?.includes('stream') && p?.label?.includes('group_feed') && p?.data?.node) {
                  const node = p.data.node;
                  // Skip non-Story nodes (e.g. GroupsSectionHeaderUnit)
                  if (node.__typename && node.__typename !== 'Story') continue;
                  const post = extractPost(node);
                  if (post) { allPosts.push(post); foundPosts = true; }
                  continue;
                }
                
                // Path 2: Standard edges array (data.node.group_feed.edges)
                const edges = p?.data?.node?.group_feed?.edges;
                if (edges && edges.length > 0) {
                  for (const edge of edges) {
                    if (allPosts.length >= maxPosts) break;
                    const node = edge?.node;
                    if (!node) continue;
                    // Skip non-Story nodes
                    if (node.__typename && node.__typename !== 'Story') continue;
                    const post = extractPost(node);
                    if (post) { allPosts.push(post); foundPosts = true; }
                  }
                }
              }
              
              // === Extract cursor from page_info ===
              for (const p of parsed) {
                // From page_info label entry
                if (p?.label?.includes('page_info') && p?.data?.page_info?.end_cursor) {
                  newCursor = p.data.page_info.end_cursor;
                  break;
                }
                // From standard group_feed.page_info
                const pi = p?.data?.node?.group_feed?.page_info;
                if (pi?.end_cursor && pi?.has_next_page !== false) {
                  newCursor = pi.end_cursor;
                  // Don't break — page_info label entry is more reliable if present
                }
              }
              
              cursor = newCursor;
              
              // No new posts found or no more pages
              if (allPosts.length === prevCount || !cursor) break;
              
              // Delay between pages to avoid rate limiting
              await new Promise(r => setTimeout(r, 200));
            }
            
            // ===== STEP 2: Fetch ALL images per post (skip for delete-only mode) =====
            if (!_skipImageFetch) for (let p = 0; p < allPosts.length; p++) {
              const post = allPosts[p];
              // Always fetch full post data for posts with images (feed only gives max 5)
              if (!post.has_video && post.media_urls.length > 0 && post.story_id) {
                try {
                  const fullPostVars = JSON.stringify({
                    feedbackSource: 2,
                    feedLocation: 'POST_PERMALINK_DIALOG',
                    focusCommentID: null,
                    privacySelectorRenderLocation: 'COMET_STREAM',
                    renderLocation: 'permalink',
                    scale: 1,
                    shouldChangeNodeFieldName: true,
                    storyID: post.story_id,
                    useDefaultActor: false
                  });
                  
                  const fullPostText = await gqlFetch(
                    '26152687281037864',
                    'CometSinglePostDialogContentQuery',
                    fullPostVars
                  );
                  
                  const fullParsed = parseMultiLineJson(fullPostText);
                  
                  // Extract ALL image URLs from all_subattachments ONLY
                  const fullUrls = [];
                  let foundSubattachments = false;
                  
                  for (const chunk of fullParsed) {
                    // Only look for all_subattachments.nodes — the definitive image list
                    function findSubattachments(obj) {
                      if (!obj || typeof obj !== 'object' || foundSubattachments) return;
                      if (Array.isArray(obj)) { obj.forEach(findSubattachments); return; }
                      
                      const subNodes = obj.all_subattachments?.nodes;
                      if (subNodes && subNodes.length > 0) {
                        foundSubattachments = true;
                        for (const sn of subNodes) {
                          const uri = sn?.media?.viewer_image?.uri 
                            || sn?.media?.image?.uri;
                          if (uri && !fullUrls.includes(uri)) fullUrls.push(uri);
                        }
                        return; // Found the definitive list, stop scanning
                      }
                      
                      for (const key of Object.keys(obj)) {
                        if (typeof obj[key] === 'object') findSubattachments(obj[key]);
                      }
                    }
                    findSubattachments(chunk);
                  }
                  
                  // Only replace if we found MORE images (avoid replacing with fewer)
                  if (fullUrls.length > post.media_urls.length) {
                    post.media_urls = fullUrls;
                  }
                  
                  await new Promise(r => setTimeout(r, 300));
                } catch(e) {
                  // Ignore errors, keep original media_urls
                }
              }
            }
            
            return { success: true, posts: allPosts, count: allPosts.length, cursor: cursor || null, has_more: !!cursor };
          } catch(err) {
            return { success: false, error: err.message };
          }
        }
      });
      
      const fetchResult = result?.[0]?.result;
      if (fetchResult?.success) {
        log(`[FETCH_POSTS] Got ${fetchResult.count} posts from group ${sourceGroupId}`);
        return fetchResult;
      }
      return { success: false, error: fetchResult?.error || 'Script failed' };
    } catch(err) {
      logError('[FETCH_POSTS] Error:', err);
      return { success: false, error: err.message };
    }
  },

  // === Download Images (service worker fetch - bypasses CORS via host_permissions) ===
  async DOWNLOAD_IMAGES(data) {
    const { urls } = data;
    if (!urls || !urls.length) return { success: true, images: [] };
    
    log(`[DOWNLOAD_IMAGES] Downloading ${urls.length} images via service worker...`);
    
    const results = [];
    for (const url of urls) {
      try {
        const resp = await fetch(url);
        if (!resp.ok) {
          log(`[DOWNLOAD_IMAGES] HTTP ${resp.status} for ${url.substring(0, 80)}`);
          results.push({ url, success: false, error: `HTTP ${resp.status}` });
          continue;
        }
        const blob = await resp.blob();
        // Convert blob to base64 in service worker (no FileReader available)
        const arrayBuffer = await blob.arrayBuffer();
        const bytes = new Uint8Array(arrayBuffer);
        let binary = '';
        for (let i = 0; i < bytes.length; i++) {
          binary += String.fromCharCode(bytes[i]);
        }
        const base64 = `data:${blob.type || 'image/jpeg'};base64,${btoa(binary)}`;
        results.push({ url, success: true, base64 });
      } catch(err) {
        log(`[DOWNLOAD_IMAGES] Error: ${err.message} for ${url.substring(0, 80)}`);
        results.push({ url, success: false, error: err.message });
      }
    }
    
    const ok = results.filter(i => i.success).length;
    log(`[DOWNLOAD_IMAGES] Downloaded ${ok}/${urls.length} images`);
    return { success: true, images: results };
  },

  // ==================== GROUP ACTIONS (9 Features) ====================

  // === Helper: Execute GraphQL on Facebook tab ===
  // All group action handlers below use this pattern internally

  // === Pending Posts Moderation ===
  async FETCH_PENDING_POSTS(data) {
    const { groupId, count = 10, cursor = null } = data;
    if (!groupId) return { success: false, error: 'No groupId' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    log(`[PENDING] Fetching pending posts for group ${groupId}, count=${count}, cursor=${cursor ? 'yes' : 'no'}`);
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, count, cursor],
        func: async (gid, maxItems, _cursor) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };

          // Relay internal vars (from captured request)
          const relayVars = {
            "__relay_internal__pv__GHLShouldChangeAdIdFieldNamerelayprovider": true,
            "__relay_internal__pv__GHLShouldChangeSponsoredDataFieldNamerelayprovider": true,
            "__relay_internal__pv__CometUFICommentActionLinksRewriteEnabledrelayprovider": false,
            "__relay_internal__pv__CometUFICommentAvatarStickerAnimatedImagerelayprovider": false,
            "__relay_internal__pv__IsWorkUserrelayprovider": false,
            "__relay_internal__pv__TestPilotShouldIncludeDemoAdUseCaserelayprovider": false,
            "__relay_internal__pv__FBReels_deprecate_short_form_video_context_gkrelayprovider": true,
            "__relay_internal__pv__FBReels_enable_view_dubbed_audio_type_gkrelayprovider": true,
            "__relay_internal__pv__CometImmersivePhotoCanUserDisable3DMotionrelayprovider": false,
            "__relay_internal__pv__WorkCometIsEmployeeGKProviderrelayprovider": false,
            "__relay_internal__pv__IsMergQAPollsrelayprovider": false,
            "__relay_internal__pv__FBReels_enable_meta_ai_label_gkrelayprovider": true,
            "__relay_internal__pv__FBReelsMediaFooter_comet_enable_reels_ads_gkrelayprovider": true,
            "__relay_internal__pv__CometUFIReactionsEnableShortNamerelayprovider": false,
            "__relay_internal__pv__CometUFIShareActionMigrationrelayprovider": true,
            "__relay_internal__pv__CometUFISingleLineUFIrelayprovider": false,
            "__relay_internal__pv__CometUFI_dedicated_comment_routable_dialog_gkrelayprovider": true,
            "__relay_internal__pv__StoriesArmadilloReplyEnabledrelayprovider": true,
            "__relay_internal__pv__FBReelsIFUTileContent_reelsIFUPlayOnHoverrelayprovider": true,
            "__relay_internal__pv__GroupsCometGYSJFeedItemHeightrelayprovider": 206,
            "__relay_internal__pv__ShouldEnableBakedInTextStoriesrelayprovider": false,
            "__relay_internal__pv__StoriesShouldIncludeFbNotesrelayprovider": false
          };

          // IMPORTANT: Use the SAME query for both initial and pagination
          // to ensure cursor compatibility. SearchSectionQuery cursors are
          // NOT compatible with FeedPaginationQuery, causing duplicate results.
          const docId = '26002878669348555';
          const friendlyName = 'GroupsCometPendingPostsFeedPaginationQuery';
          const vars = {
            count: 3, cursor: _cursor || null,
            feedLocation: 'GROUP_PENDING', feedbackSource: 0,
            focusCommentID: null, hoistedPostID: null,
            pendingStoriesOrderBy: null,
            privacySelectorRenderLocation: 'COMET_STREAM',
            renderLocation: 'group_pending_queue', scale: 1,
            useDefaultActor: false, id: gid,
            ...relayVars
          };

          // Helper: execute GraphQL fetch
          async function fetchGraphQL(docId, friendlyName, vars) {
            const p = new URLSearchParams({
              av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd,
              fb_api_caller_class: 'RelayModern',
              fb_api_req_friendly_name: friendlyName,
              server_timestamps: 'true',
              doc_id: docId,
              variables: JSON.stringify(vars)
            });
            const r = await fetch('https://www.facebook.com/api/graphql/', {
              method: 'POST',
              headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
              body: p.toString(), credentials: 'include'
            });
            const text = await r.text();
            const cleaned = text.replace(/^for \(;;\);?/, '').trim();
            const lines = cleaned.split('\n').filter(l => l.trim());
            const parsed = [];
            for (const line of lines) {
              try { parsed.push(JSON.parse(line)); } catch(e) {}
            }
            return parsed;
          }

          let parsed = await fetchGraphQL(docId, friendlyName, vars);

          // If pagination returned errors or no useful data, try fallback queries
          if (_cursor && parsed.length > 0) {
            const hasErrors = parsed.some(c => c?.errors?.length > 0);
            const hasEdges = parsed.some(c => {
              return c?.data?.group?.pending_posts_search?.edges?.length > 0
                || c?.data?.node?.pending_posts_section_stories?.edges?.length > 0
                || c?.data?.group?.group_pending_queue?.edges?.length > 0
                || c?.data?.node?.group_pending_queue?.edges?.length > 0
                || (c?.label && c?.data?.node?.__typename === 'Story');
            });
            if (hasErrors && !hasEdges) {
              console.log('[PENDING] FeedPaginationQuery returned errors, trying SearchSectionQuery fallback...');
              // Fallback: use initial SearchSectionQuery with cursor injected
              const fallbackVars = {
                afterTime: 0, authorID: null, beforeTime: 0,
                groupID: gid, linkType: null, orderBy: null,
                postType: null, search: null, source: null,
                subgroupIDs: null, feedbackSource: 0,
                feedLocation: 'GROUP_PENDING',
                privacySelectorRenderLocation: 'COMET_STREAM',
                renderLocation: 'group_pending_queue', scale: 1,
                cursor: _cursor, count: 3,
                ...relayVars
              };
              parsed = await fetchGraphQL('26569990999270187', 'GroupsCometPendingPostsPostsSearchSectionQuery', fallbackVars);

              // If still no data, try yet another approach — GroupCometPendingPostsCardListPaginationQuery
              const hasEdges2 = parsed.some(c => {
                return c?.data?.group?.pending_posts_search?.edges?.length > 0
                  || c?.data?.node?.pending_posts_section_stories?.edges?.length > 0
                  || (c?.label && c?.data?.node?.__typename === 'Story');
              });
              if (!hasEdges2) {
                console.log('[PENDING] SearchSectionQuery fallback also failed, trying direct group query...');
                const directVars = {
                  id: gid, count: 3, cursor: _cursor,
                  feedLocation: 'GROUP_PENDING', feedbackSource: 0,
                  privacySelectorRenderLocation: 'COMET_STREAM',
                  renderLocation: 'group_pending_queue', scale: 1,
                  useDefaultActor: false,
                  ...relayVars
                };
                // Try with the pagination doc_id but simpler vars
                parsed = await fetchGraphQL('26002878669348555', 'GroupsCometPendingPostsFeedPaginationQuery', directVars);
              }
            }
          }

          // DEBUG: Log response structure
          console.log('[PENDING DEBUG] Parsed chunks:', parsed.length);
          for (let ci = 0; ci < Math.min(parsed.length, 5); ci++) {
            const c = parsed[ci];
            const topKeys = Object.keys(c || {});
            console.log(`[PENDING DEBUG] Chunk ${ci} keys:`, topKeys.join(', '));
            if (c?.data) {
              const dataKeys = Object.keys(c.data);
              console.log(`[PENDING DEBUG] Chunk ${ci} data keys:`, dataKeys.join(', '));
              if (c.data.group) console.log(`[PENDING DEBUG] group keys:`, Object.keys(c.data.group).join(', '));
              if (c.data.node) console.log(`[PENDING DEBUG] node keys:`, Object.keys(c.data.node).join(', '));
            }
            if (c?.label) console.log(`[PENDING DEBUG] Chunk ${ci} label:`, c.label);
            if (c?.errors) console.log(`[PENDING DEBUG] Chunk ${ci} errors:`, c.errors.map(e => e?.message || JSON.stringify(e).slice(0,100)));
          }

          // Helper
          function getVal(obj, path) {
            return path.split('.').reduce((o, k) => {
              if (!o) return undefined;
              const m = k.match(/^(.+?)\[(\d+)\]$/);
              return m ? o[m[1]]?.[Number(m[2])] : o[k];
            }, obj);
          }

          const stories = [];
          const seenIds = new Set();
          let newCursor = null;
          let hasNextPage = false;

          for (const chunk of parsed) {
            const edgePaths = [
              chunk?.data?.group?.pending_posts_search?.edges,
              chunk?.data?.node?.pending_posts_section_stories?.edges,
              chunk?.data?.group?.group_pending_queue?.edges,
              chunk?.data?.node?.group_pending_queue?.edges,
              chunk?.data?.group?.pending_posts?.edges,
              chunk?.data?.node?.pending_posts?.edges
            ];

            for (const edges of edgePaths) {
              if (!edges || !edges.length) continue;
              for (const edge of edges) {
                const node = edge?.node;
                if (!node || !node.id) continue;
                if (seenIds.has(node.id)) continue;
                seenIds.add(node.id);

                const authorName = getVal(node, 'comet_sections.context_layout.story.comet_sections.actor_photo.story.actors[0].name')
                  || node.actors?.[0]?.name || 'Unknown';
                const authorId = getVal(node, 'comet_sections.context_layout.story.comet_sections.actor_photo.story.actors[0].id')
                  || node.actors?.[0]?.id || '';
                const message = getVal(node, 'comet_sections.content.story.message.text')
                  || getVal(node, 'comet_sections.content.story.comet_sections.message.story.message.text')
                  || node.message?.text || '';
                const timestamp = getVal(node, 'comet_sections.timestamp.story.creation_time')
                  || node.created_time || node.creation_time || null;

                // Extract media - real API shows media in attached_story.attachments[]
                const mediaUrls = [];
                const attachmentsPaths = [
                  getVal(node, 'attached_story.attachments'),
                  getVal(node, 'attached_story.comet_sections.message_container.story.attachments'),
                  getVal(node, 'comet_sections.content.story.attached_story.attachments'),
                  node.attachments
                ];
                for (const attachments of attachmentsPaths) {
                  if (!attachments || !Array.isArray(attachments)) continue;
                  for (const att of attachments) {
                    const styles = att?.styles?.attachment || att;
                    const media = styles?.media || styles?.all_subattachments?.nodes?.[0]?.media;
                    const uri = media?.thumbnailImage?.uri || media?.viewer_image?.uri || media?.photo_image?.uri || media?.image?.uri || media?.animated_image?.uri;
                    if (uri && !mediaUrls.includes(uri)) mediaUrls.push(uri);
                  }
                  if (mediaUrls.length > 0) break; // Found media, stop checking other paths
                }

                // Detect reshare: multiple signals
                const attachedStory = getVal(node, 'attached_story') || getVal(node, 'comet_sections.content.story.attached_story');
                const attachedStoryType = attachedStory?.__typename;
                const storyAttachStyle = getVal(node, 'comet_sections.content.story.story_attachment_style')
                  || getVal(node, 'story_attachment_style');
                const hasAttachedStoryLayout = !!getVal(node, 'comet_sections.attached_story_layout');
                const attachedHasStoryProps = !!(attachedStory && (attachedStory.actors || attachedStory.comet_sections || attachedStory.message || attachedStory.url));
                
                const isReshare = !!(
                  attachedStoryType === 'Story' ||
                  storyAttachStyle === 'SHARE' ||
                  storyAttachStyle === 'RESHARE' ||
                  hasAttachedStoryLayout ||
                  attachedHasStoryProps
                );
                
                console.log(`[PENDING] Post ${node.id}: msg="${(message || '').slice(0,30)}", attachedType=${attachedStoryType}, attachStyle=${storyAttachStyle}, hasLayout=${hasAttachedStoryLayout}, hasProps=${attachedHasStoryProps}, isReshare=${isReshare}, media=${mediaUrls.length}`);
                
                const hasVideo = attachmentsPaths.some(atts => {
                  if (!atts || !Array.isArray(atts)) return false;
                  return atts.some(a => {
                    const t = (a?.styles?.attachment || a)?.media?.__typename;
                    return t === 'Video' || t === 'LiveVideo';
                  });
                });
                
                // Determine post type
                const postType = isReshare ? 'reshare' : hasVideo ? 'video' : mediaUrls.length > 0 ? 'image' : 'text';
                
                stories.push({
                  storyId: node.id,
                  authorName,
                  authorId,
                  message,
                  createdTime: timestamp,
                  mediaUrls,
                  hasMedia: mediaUrls.length > 0,
                  hasVideo,
                  isReshare,
                  postType
                });
              }
            }

            // Also handle streaming label entries with individual Story nodes
            if (chunk?.label && chunk?.data?.node && chunk.data.node.__typename === 'Story' && !seenIds.has(chunk.data.node.id)) {
              const node = chunk.data.node;
              seenIds.add(node.id);
              const authorName = getVal(node, 'comet_sections.context_layout.story.comet_sections.actor_photo.story.actors[0].name') || 'Unknown';
              const authorId = getVal(node, 'comet_sections.context_layout.story.comet_sections.actor_photo.story.actors[0].id') || '';
              const message = getVal(node, 'comet_sections.content.story.message.text') || '';
              const timestamp = getVal(node, 'comet_sections.timestamp.story.creation_time') || null;
              
              // Extract media for streaming nodes too
              const mediaUrls = [];
              const attachmentsPaths = [
                getVal(node, 'attached_story.attachments'),
                getVal(node, 'comet_sections.content.story.attached_story.attachments'),
                node.attachments
              ];
              for (const attachments of attachmentsPaths) {
                if (!attachments || !Array.isArray(attachments)) continue;
                for (const att of attachments) {
                  const styles = att?.styles?.attachment || att;
                  const media = styles?.media || styles?.all_subattachments?.nodes?.[0]?.media;
                  const uri = media?.thumbnailImage?.uri || media?.viewer_image?.uri || media?.photo_image?.uri || media?.image?.uri;
                  if (uri && !mediaUrls.includes(uri)) mediaUrls.push(uri);
                }
                if (mediaUrls.length > 0) break;
              }
              
              const sAttached = getVal(node, 'attached_story') || getVal(node, 'comet_sections.content.story.attached_story');
              const sIsReshare = !!(sAttached && (sAttached.actors || sAttached.comet_sections || sAttached.message));
              const sPostType = sIsReshare ? 'reshare' : mediaUrls.length > 0 ? 'image' : 'text';
              stories.push({ storyId: node.id, authorName, authorId, message, createdTime: timestamp, mediaUrls, hasMedia: mediaUrls.length > 0, hasVideo: false, isReshare: sIsReshare, postType: sPostType });
            }

            // Extract cursor from page_info
            const piPaths = [
              chunk?.data?.group?.pending_posts_search?.page_info,
              chunk?.data?.node?.pending_posts_section_stories?.page_info,
              chunk?.data?.group?.group_pending_queue?.page_info,
              chunk?.data?.node?.group_pending_queue?.page_info,
              chunk?.data?.group?.pending_posts?.page_info,
              chunk?.data?.node?.pending_posts?.page_info,
              chunk?.data?.page_info
            ];
            for (const pi of piPaths) {
              if (pi?.end_cursor) {
                newCursor = pi.end_cursor;
                hasNextPage = pi.has_next_page !== false;
                break;
              }
            }
            if (chunk?.label?.includes('page_info') && chunk?.data?.page_info?.end_cursor) {
              newCursor = chunk.data.page_info.end_cursor;
              hasNextPage = chunk.data.page_info.has_next_page !== false;
            }
          }

          console.log(`[PENDING] Found ${stories.length} stories, cursor: ${newCursor ? 'yes' : 'no'}, hasNext: ${hasNextPage}`);

          return { success: true, stories, total: stories.length, cursor: newCursor, has_more: hasNextPage && !!newCursor };
        }
      });
      const result = results?.[0]?.result || { success: false, error: 'Script failed' };
      if (result.success) {
        log(`[PENDING] Got ${result.total} pending posts`);
      }
      return result;
    } catch (e) { return { success: false, error: e.message }; }
  },

  async APPROVE_PENDING_POST(data) {
    const { groupId, storyId } = data;
    if (!groupId || !storyId) return { success: false, error: 'Missing groupId or storyId' };
    log(`[APPROVE] Approving story ${storyId} in group ${groupId}`);
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, storyId],
        func: async (gid, sid) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };

          console.log(`[APPROVE] userId=${userId}, sid=${sid}, gid=${gid}`);
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '9662498830453426', variables: JSON.stringify({ input: { action_source: 'GROUP_PENDING_POSTS', group_id: gid, story_id: sid, actor_id: userId, client_mutation_id: String(Date.now()) } }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          console.log(`[APPROVE] Response status=${r.status}, body=${t.slice(0, 500)}`);
          const c = t.replace(/^for \(;;\);?/, '').trim();
          try {
            const json = JSON.parse(c.split('\n')[0]);
            if (json?.data) return { success: true, storyId: sid };
            return { success: false, error: json?.errors?.[0]?.message || 'Unknown error', raw: c.slice(0, 300) };
          } catch (parseErr) {
            return { success: false, error: 'Parse error', raw: c.slice(0, 300) };
          }
        }
      });
      const result = results?.[0]?.result || { success: false, error: 'Script failed' };
      log(`[APPROVE] Result:`, JSON.stringify(result));
      return result;
    } catch (e) { return { success: false, error: e.message }; }
  },

  async DECLINE_PENDING_POST(data) {
    const { groupId, storyId, memberId } = data;
    if (!groupId || !storyId) return { success: false, error: 'Missing groupId or storyId' };
    log(`[DECLINE] Declining story ${storyId} in group ${groupId}, member=${memberId || 'unknown'}`);
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, storyId, memberId],
        func: async (gid, sid, mid) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };

          console.log(`[DECLINE] userId=${userId}, sid=${sid}, gid=${gid}, mid=${mid}`);
          const p = new URLSearchParams({
            av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd,
            fb_api_caller_class: 'RelayModern',
            fb_api_req_friendly_name: 'GroupsCometDeclinePendingStoryMutation',
            server_timestamps: 'true',
            doc_id: '24257178983980731',
            variables: JSON.stringify({
              currentSection: 'PENDING_POSTS',
              input: {
                action_source: 'GROUP_PENDING_POSTS',
                group_id: gid,
                story_id: sid,
                actor_id: userId,
                client_mutation_id: String(Date.now())
              },
              memberID: mid || userId,
              scale: 1
            })
          });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          console.log(`[DECLINE] Response status=${r.status}, body=${t.slice(0, 500)}`);
          const c = t.replace(/^for \(;;\);?/, '').trim();
          try {
            const json = JSON.parse(c.split('\n')[0]);
            if (json?.data) return { success: true, storyId: sid };
            return { success: false, error: json?.errors?.[0]?.message || 'Unknown error', raw: c.slice(0, 300) };
          } catch (parseErr) {
            return { success: false, error: 'Parse error', raw: c.slice(0, 300) };
          }
        }
      });
      const result = results?.[0]?.result || { success: false, error: 'Script failed' };
      log(`[DECLINE] Result:`, JSON.stringify(result));
      return result;
    } catch (e) { return { success: false, error: e.message }; }
  },

  async BULK_MODERATE_POSTS(data) {
    const { groupId, items = [], delayMs = 1500 } = data;
    if (!groupId || !items.length) return { success: false, error: 'Missing data' };
    const results = { total: items.length, success: 0, failed: 0, items: [] };
    for (let i = 0; i < items.length; i++) {
      const item = items[i];
      const handler = item.action === 'approve' ? 'APPROVE_PENDING_POST' : 'DECLINE_PENDING_POST';
      const result = await handlers[handler]({ groupId, storyId: item.storyId, memberId: item.memberId });
      results.items.push({ ...result, action: item.action });
      if (result.success) results.success++; else results.failed++;
      broadcast({ type: 'MODERATE_PROGRESS', data: { current: i + 1, total: items.length, ...result } });
      if (i < items.length - 1) await new Promise(r => setTimeout(r, delayMs));
    }
    return { success: true, ...results };
  },

  // === Delete Posts ===
  async DELETE_POSTS_ADMIN(data) {
    const { groupId, storyIds = [], delayMs = 2000, concurrency = 3 } = data;
    if (!groupId || !storyIds.length) return { success: false, error: 'Missing data' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    const results = { total: storyIds.length, successCount: 0, failed: 0, items: [] };
    let completed = 0;

    // Delete a single post with fallback doc_ids
    const deleteOne = async (sid) => {
      // doc_ids: primary = group_content_remove, fallback = feed story delete
      const docIds = ['24487184117551290', '3967628089995602'];
      for (const docId of docIds) {
        try {
          const r = await chrome.scripting.executeScript({
            target: { tabId: tab.id }, world: 'MAIN', args: [groupId, sid, docId],
            func: async (gid, storyId, did) => {
              let userId, dtsg, lsd = '';
              if (typeof require === 'function') {
                try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
                try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
                try { lsd = require('LSD')?.token || ''; } catch(e) {}
              }
              if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
              if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
              const vars = { input: { client_mutation_id: String(Date.now()), actor_id: userId, admin_notes: '', group_id: gid, selected_rules: [], send_warning: false, share_feedback: false, source: 'group_mall', story_id: storyId }, profileID: userId };
              // For feed story delete mutation use different variable structure
              if (did === '3967628089995602') {
                vars.input = { story_id: storyId, source: 'group_mall', share_feedback: false, group_id: gid, selected_rules: [], client_mutation_id: String(Date.now()), admin_notes: '', actor_id: userId };
                vars.post_id = storyId;
              }
              const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: did, variables: JSON.stringify(vars) });
              const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
              const t = await r.text();
              const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
              const deleted = json?.data?.story_delete?.deleted_story_id || json?.data?.group_content_remove?.story?.id;
              return deleted ? { success: true } : { success: false, error: json?.errors?.[0]?.message || 'Failed' };
            }
          });
          const result = r?.[0]?.result || { success: false, error: 'Script failed' };
          if (result.success) return result;
          // If this doc_id failed, try next
        } catch (e) {
          // Try next doc_id
        }
      }
      return { success: false, error: 'All delete methods failed' };
    };

    // Process with concurrency (parallel workers)
    const queue = [...storyIds];
    let index = 0;
    const worker = async () => {
      while (index < queue.length) {
        const i = index++;
        const sid = queue[i];
        const result = await deleteOne(sid);
        results.items.push({ storyId: sid, ...result });
        if (result.success) results.successCount++; else results.failed++;
        completed++;
        broadcast({ type: 'DELETE_POST_PROGRESS', data: { current: completed, total: storyIds.length } });
        if (index < queue.length) await new Promise(r => setTimeout(r, delayMs));
      }
    };
    const workers = Array.from({ length: Math.min(concurrency, storyIds.length) }, () => worker());
    await Promise.all(workers);
    return { success: true, ...results };
  },

  // === Spam Moderation ===
  async FETCH_SPAM_QUEUE(data) {
    const { groupId, count = 10, cursor = null } = data;
    if (!groupId) return { success: false, error: 'No groupId' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    log(`[SPAM] Fetching spam queue for group ${groupId}, count=${count}`);
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, count, cursor],
        func: async (gid, maxItems, _cursor) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          
          // Relay vars from captured real request
          const relayVars = '"__relay_internal__pv__GHLShouldChangeAdIdFieldNamerelayprovider":true,"__relay_internal__pv__GHLShouldChangeSponsoredDataFieldNamerelayprovider":true,"__relay_internal__pv__CometUFICommentActionLinksRewriteEnabledrelayprovider":false,"__relay_internal__pv__CometUFICommentAvatarStickerAnimatedImagerelayprovider":false,"__relay_internal__pv__IsWorkUserrelayprovider":false,"__relay_internal__pv__TestPilotShouldIncludeDemoAdUseCaserelayprovider":false,"__relay_internal__pv__FBReels_deprecate_short_form_video_context_gkrelayprovider":true,"__relay_internal__pv__FBReels_enable_view_dubbed_audio_type_gkrelayprovider":true,"__relay_internal__pv__CometImmersivePhotoCanUserDisable3DMotionrelayprovider":false,"__relay_internal__pv__WorkCometIsEmployeeGKProviderrelayprovider":false,"__relay_internal__pv__IsMergQAPollsrelayprovider":false,"__relay_internal__pv__FBReels_enable_meta_ai_label_gkrelayprovider":true,"__relay_internal__pv__FBReelsMediaFooter_comet_enable_reels_ads_gkrelayprovider":true,"__relay_internal__pv__CometUFIReactionsEnableShortNamerelayprovider":false,"__relay_internal__pv__CometUFIShareActionMigrationrelayprovider":true,"__relay_internal__pv__CometUFISingleLineUFIrelayprovider":false,"__relay_internal__pv__CometUFI_dedicated_comment_routable_dialog_gkrelayprovider":false,"__relay_internal__pv__StoriesArmadilloReplyEnabledrelayprovider":true,"__relay_internal__pv__FBReelsIFUTileContent_reelsIFUPlayOnHoverrelayprovider":true,"__relay_internal__pv__GroupsCometGYSJFeedItemHeightrelayprovider":206,"__relay_internal__pv__ShouldEnableBakedInTextStoriesrelayprovider":false,"__relay_internal__pv__StoriesShouldIncludeFbNotesrelayprovider":false';
          
          const vars = JSON.stringify({
            contentType: null, count: maxItems, cursor: _cursor,
            feedLocation: 'GROUPS_MODMIN_REVIEW_FOLDER', feedbackSource: 0,
            focusCommentID: null, privacySelectorRenderLocation: 'COMET_STREAM',
            renderLocation: 'groups_modmin_review_folder', scale: 1,
            searchTerm: null, useDefaultActor: false, id: gid,
            ...JSON.parse('{' + relayVars + '}')
          });
          
          const p = new URLSearchParams({
            av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd,
            fb_api_caller_class: 'RelayModern',
            fb_api_req_friendly_name: 'GroupsCometModminReviewFolderContentContainerQuery',
            server_timestamps: 'true',
            doc_id: '26362784449986269', variables: vars
          });
          
          const r = await fetch('https://www.facebook.com/api/graphql/', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: p.toString(), credentials: 'include'
          });
          const text = await r.text();
          const cleaned = text.replace(/^for \(;;\);?/, '').trim();
          
          // Parse multi-line streaming response
          const lines = cleaned.split('\n').filter(l => l.trim());
          const parsed = [];
          for (const line of lines) {
            try { parsed.push(JSON.parse(line)); } catch(e) {}
          }
          
          // Helper to get nested value
          function getVal(obj, path) {
            return path.split('.').reduce((o, k) => {
              if (!o) return undefined;
              const m = k.match(/^(.+?)\[(\d+)\]$/);
              return m ? o[m[1]]?.[Number(m[2])] : o[k];
            }, obj);
          }
          
          const items = [];
          const seenIds = new Set();
          let newCursor = null;
          
          for (const chunk of parsed) {
            // Extract items from edges (standard or streaming)
            const edgePaths = [
              chunk?.data?.node?.group_moderation_queue_tab_card?.review_items?.edges,
              chunk?.data?.node?.modmin_review_folder?.edges,
              chunk?.data?.group?.modmin_review_folder?.edges
            ];
            
            for (const edges of edgePaths) {
              if (!edges || !edges.length) continue;
              for (const edge of edges) {
                const node = edge?.node;
                if (!node || !node.id) continue;
                if (seenIds.has(node.id)) continue;
                seenIds.add(node.id);
                
                // Extract story data
                const authorName = getVal(node, 'comet_sections.context_layout.story.comet_sections.actor_photo.story.actors[0].name')
                  || node.actors?.[0]?.name || 'Unknown';
                const authorId = getVal(node, 'comet_sections.context_layout.story.comet_sections.actor_photo.story.actors[0].id')
                  || node.actors?.[0]?.id || '';
                const message = getVal(node, 'comet_sections.content.story.message.text')
                  || getVal(node, 'comet_sections.content.story.comet_sections.message.story.message.text')
                  || node.message?.text || '';
                const timestamp = getVal(node, 'comet_sections.timestamp.story.creation_time')
                  || node.created_time || null;
                
                // Extract media
                const mediaUrls = [];
                const attachments = node.attachments || [];
                for (const att of attachments) {
                  const styles = att?.styles?.attachment || att;
                  const uri = styles?.media?.viewer_image?.uri || styles?.media?.photo_image?.uri || styles?.media?.image?.uri;
                  if (uri && !mediaUrls.includes(uri)) mediaUrls.push(uri);
                }
                
                items.push({
                  storyId: node.id,
                  authorName,
                  authorId,
                  message,
                  contentType: node.__typename || 'Story',
                  createdTime: timestamp,
                  mediaUrls,
                  hasVideo: attachments.some(a => {
                    const t = (a?.styles?.attachment || a)?.media?.__typename;
                    return t === 'Video' || t === 'LiveVideo';
                  })
                });
              }
            }
            
            // Also check streaming label entries
            if (chunk?.label && chunk?.data?.node && !seenIds.has(chunk.data.node.id)) {
              const node = chunk.data.node;
              if (node.__typename === 'Story' && node.id) {
                seenIds.add(node.id);
                const authorName = getVal(node, 'comet_sections.context_layout.story.comet_sections.actor_photo.story.actors[0].name') || 'Unknown';
                const authorId = getVal(node, 'comet_sections.context_layout.story.comet_sections.actor_photo.story.actors[0].id') || '';
                const message = getVal(node, 'comet_sections.content.story.message.text') || '';
                const timestamp = getVal(node, 'comet_sections.timestamp.story.creation_time') || null;
                items.push({ storyId: node.id, authorName, authorId, message, contentType: 'Story', createdTime: timestamp, mediaUrls: [], hasVideo: false });
              }
            }
            
            // Extract cursor
            const piPaths = [
              chunk?.data?.node?.group_moderation_queue_tab_card?.review_items?.page_info,
              chunk?.data?.node?.modmin_review_folder?.page_info,
              chunk?.data?.group?.modmin_review_folder?.page_info,
              chunk?.data?.page_info
            ];
            for (const pi of piPaths) {
              if (pi?.end_cursor) { newCursor = pi.end_cursor; break; }
            }
            // page_info from labeled chunk  
            if (chunk?.label?.includes('page_info') && chunk?.data?.page_info?.end_cursor) {
              newCursor = chunk.data.page_info.end_cursor;
            }
          }
          
          return { success: true, items, total: items.length, cursor: newCursor, has_more: !!newCursor };
        }
      });
      const result = results?.[0]?.result || { success: false, error: 'Script failed' };
      if (result.success) log(`[SPAM] Got ${result.total} spam items`);
      return result;
    } catch (e) { return { success: false, error: e.message }; }
  },

  async MODERATE_SPAM(data) {
    const { groupId, items = [], delayMs = 1500 } = data;
    if (!groupId || !items.length) return { success: false, error: 'Missing data' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    const results = { total: items.length, success: 0, failed: 0, items: [] };
    for (let i = 0; i < items.length; i++) {
      const item = items[i];
      const docId = item.action === 'approve' ? '9662498830453426' : '30216535437945304';
      try {
        const r = await chrome.scripting.executeScript({
          target: { tabId: tab.id }, world: 'MAIN', args: [groupId, item.storyId, item.memberId, docId, item.action],
          func: async (gid, sid, mid, did, action) => {
            let userId, dtsg, lsd = '';
            if (typeof require === 'function') {
              try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
              try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
              try { lsd = require('LSD')?.token || ''; } catch(e) {}
            }
            if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
            if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
            const vars = action === 'approve'
              ? { input: { action_source: 'GROUP_MODMIN_REVIEW_FOLDER', group_id: gid, story_id: sid, actor_id: userId, client_mutation_id: String(Date.now()) } }
              : { input: { action_source: 'GROUP_MODMIN_REVIEW_FOLDER', group_id: gid, story_id: sid, actor_id: userId, client_mutation_id: String(Date.now()) }, contentType: 'GROUP_POST', member_id: mid || userId };
            const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: did, variables: JSON.stringify(vars) });
            const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
            const t = await r.text();
            const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
            return json?.data ? { success: true } : { success: false, error: 'Failed' };
          }
        });
        const result = r?.[0]?.result || { success: false, error: 'Script failed' };
        results.items.push({ storyId: item.storyId, ...result });
        if (result.success) results.success++; else results.failed++;
      } catch(e) { results.items.push({ storyId: item.storyId, success: false, error: e.message }); results.failed++; }
      broadcast({ type: 'SPAM_MODERATE_PROGRESS', data: { current: i + 1, total: items.length } });
      if (i < items.length - 1) await new Promise(r => setTimeout(r, delayMs));
    }
    return { success: true, ...results };
  },

  // === Invite Friends ===
  async SEARCH_INVITABLE_FRIENDS(data) {
    const { groupId, keyword = '', cursor = null } = data;
    if (!groupId) return { success: false, error: 'No groupId' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, keyword, cursor],
        func: async (gid, kw, cur) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const docId = kw ? '24247827701489068' : '23947758888181572';
          const vars = kw
            ? { after_cursor: cur, count: 20, name: kw, scale: 1, id: gid }
            : { after_cursor: cur, category_source_id: null, category_type: 'ALL', count: 20, keyword: '', scale: 1, selected_group_ids: null, id: gid };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: docId, variables: JSON.stringify(vars) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const c = t.replace(/^for \(;;\);?/, '').trim();
          const json = JSON.parse(c.split('\n')[0]);
          const edges = json?.data?.node?.invite_friends?.edges || json?.data?.node?.categorized_invite_search?.edges || [];
          const friends = edges.map(e => ({ id: e.node?.id, name: e.node?.name, avatar: e.node?.profile_picture?.uri || e.node?.profilePicLarge?.uri, invited: e.node?.is_invited || false }));
          return { success: true, friends, total: friends.length };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  async INVITE_FRIENDS(data) {
    const { groupId, userIds = [] } = data;
    if (!groupId || !userIds.length) return { success: false, error: 'Missing data' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, userIds],
        func: async (gid, uids) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '24974699228877844', variables: JSON.stringify({ input: { email_addresses: [], group_id: gid, source: 'comet_invite_friends', user_ids: uids, actor_id: userId, client_mutation_id: String(Date.now()) }, groupID: gid }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
          return json?.data ? { success: true, invited: uids.length } : { success: false, error: json?.errors?.[0]?.message || 'Failed' };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  // === Block Members ===
  async BLOCK_MEMBERS(data) {
    const { groupId, userIds = [], options = {}, delayMs = 2000 } = data;
    if (!groupId || !userIds.length) return { success: false, error: 'Missing data' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    const results = { total: userIds.length, success: 0, failed: 0, items: [] };
    for (let i = 0; i < userIds.length; i++) {
      try {
        const r = await chrome.scripting.executeScript({
          target: { tabId: tab.id }, world: 'MAIN', args: [groupId, userIds[i], options],
          func: async (gid, uid, opts) => {
            let userId, dtsg, lsd = '';
            if (typeof require === 'function') {
              try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
              try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
              try { lsd = require('LSD')?.token || ''; } catch(e) {}
            }
            if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
            if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
            const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '33570111005967600', variables: JSON.stringify({ groupID: gid, input: { client_mutation_id: String(Date.now()), actor_id: userId, action_source: 'CONTEXTUAL_PROFILE', apply_to_later_created_accounts: opts.applyToLater || false, apply_to_other_groups_you_manage: opts.applyToOtherGroups || false, delete_recent_comments: opts.deleteComments || false, delete_recent_invites: opts.deleteInvites || false, delete_recent_poll_options: false, delete_recent_posts: opts.deletePosts || false, delete_recent_reactions: false, delete_recent_story_threads: false, group_id: gid, user_id: uid }, inviteShortLinkKey: null, memberID: uid, scale: 1 }) });
            const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
            const t = await r.text();
            const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
            return json?.data ? { success: true } : { success: false, error: json?.errors?.[0]?.message || 'Failed' };
          }
        });
        const result = r?.[0]?.result || { success: false, error: 'Script failed' };
        results.items.push({ userId: userIds[i], ...result });
        if (result.success) results.success++; else results.failed++;
      } catch(e) { results.items.push({ userId: userIds[i], success: false, error: e.message }); results.failed++; }
      broadcast({ type: 'BLOCK_PROGRESS', data: { current: i + 1, total: userIds.length } });
      if (i < userIds.length - 1) await new Promise(r => setTimeout(r, delayMs));
    }
    return { success: true, ...results };
  },

  // === Admin Management ===
  async FETCH_GROUP_MEMBERS(data) {
    const { groupId, membershipType = 'MEMBER', cursor = null, count = 20 } = data;
    if (!groupId) return { success: false, error: 'No groupId' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, membershipType, cursor, count],
        func: async (gid, mType, cur, cnt) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '25639273902400228', variables: JSON.stringify({ count: cnt, cursor: cur, groupID: gid, membershipType: mType, scale: 1, search: null, statusStaticFilter: null, id: gid }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const c = t.replace(/^for \(;;\);?/, '').trim();
          const json = JSON.parse(c.split('\n')[0]);
          const edges = json?.data?.node?.new_members?.edges || json?.data?.node?.profiles?.edges || [];
          const members = edges.map(e => ({ id: e.node?.id, name: e.node?.name, avatar: e.node?.profile_picture?.uri, role: e.node?.membership?.association_type || 'MEMBER', joinedDate: e.node?.membership?.join_status_text?.text }));
          const pageInfo = json?.data?.node?.new_members?.page_info || json?.data?.node?.profiles?.page_info || {};
          return { success: true, members, hasNextPage: pageInfo.has_next_page, endCursor: pageInfo.end_cursor };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  async FETCH_GROUP_ADMINS(data) {
    const { groupId, cursor = null } = data;
    if (!groupId) return { success: false, error: 'No groupId' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, cursor],
        func: async (gid, cur) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '25883864567968200', variables: JSON.stringify({ count: 20, cursor: cur, groupID: gid, scale: 1, id: gid }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const c = t.replace(/^for \(;;\);?/, '').trim();
          const json = JSON.parse(c.split('\n')[0]);
          const edges = json?.data?.node?.admins_and_moderators?.edges || [];
          const admins = edges.map(e => ({ id: e.node?.id, name: e.node?.name, avatar: e.node?.profile_picture?.uri, role: e.node?.membership?.association_type || 'ADMIN' }));
          const pageInfo = json?.data?.node?.admins_and_moderators?.page_info || {};
          return { success: true, admins, hasNextPage: pageInfo.has_next_page, endCursor: pageInfo.end_cursor };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  async ADD_ADMIN(data) {
    const { groupId, userId: targetUserId, adminType = 'admin' } = data;
    if (!groupId || !targetUserId) return { success: false, error: 'Missing data' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, targetUserId, adminType],
        func: async (gid, tuid, aType) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '26109778708640692', variables: JSON.stringify({ groupID: gid, memberID: tuid, input: { action_source: 'MEMBER_LIST', admin_type: aType, group_id: gid, user_id: tuid, actor_id: userId, client_mutation_id: String(Date.now()) }, scale: 1, isContextualProfile: false }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
          return json?.data ? { success: true } : { success: false, error: json?.errors?.[0]?.message || 'Failed' };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  async REMOVE_ADMIN(data) {
    const { groupId, userId: targetUserId, adminType = 'admin' } = data;
    if (!groupId || !targetUserId) return { success: false, error: 'Missing data' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, targetUserId, adminType],
        func: async (gid, tuid, aType) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '25447572011592270', variables: JSON.stringify({ groupID: gid, memberID: tuid, input: { admin_type: aType, group_id: gid, user_id: tuid, actor_id: userId, client_mutation_id: String(Date.now()) }, scale: 1, isContextualProfile: false }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
          return json?.data ? { success: true } : { success: false, error: json?.errors?.[0]?.message || 'Failed' };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  // === Auto Comment ===
  async CREATE_COMMENT(data) {
    const { groupId, feedbackId, text } = data;
    if (!feedbackId || !text) return { success: false, error: 'Missing feedbackId or text' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, feedbackId, text],
        func: async (gid, fid, txt) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '25784859391141468', variables: JSON.stringify({ feedLocation: 'GROUP', feedbackSource: 0, groupID: gid || '', input: { client_mutation_id: String(Date.now()), actor_id: userId, attachments: null, feedback_id: fid, formatting_style: null, message: { ranges: [], text: txt }, feedback_source: 'PROFILE', idempotence_token: `client:${crypto.randomUUID()}`, session_id: crypto.randomUUID() }, inviteShortLinkKey: null, renderLocation: null, scale: 1, useDefaultActor: false, focusCommentID: null }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
          return json?.data?.comment_create ? { success: true } : { success: false, error: json?.errors?.[0]?.message || 'Failed' };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  async BULK_COMMENT(data) {
    const { posts = [], text, delayMs = 5000 } = data;
    if (!posts.length || !text) return { success: false, error: 'Missing data' };
    const results = { total: posts.length, success: 0, failed: 0, items: [] };
    for (let i = 0; i < posts.length; i++) {
      const post = posts[i];
      const result = await handlers.CREATE_COMMENT({ groupId: post.groupId, feedbackId: post.feedbackId, text });
      results.items.push({ postId: post.postId, ...result });
      if (result.success) results.success++; else results.failed++;
      broadcast({ type: 'COMMENT_PROGRESS', data: { current: i + 1, total: posts.length, ...result } });
      if (i < posts.length - 1) await new Promise(r => setTimeout(r, delayMs));
    }
    return { success: true, ...results };
  },

  // === Pin/Announcement ===
  async FETCH_HIGHLIGHTS(data) {
    const { groupId } = data;
    if (!groupId) return { success: false, error: 'No groupId' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId],
        func: async (gid) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '26295402700052132', variables: JSON.stringify({ count: 20, cursor: null, privacySelectorRenderLocation: 'COMET_STREAM', renderLocation: 'group', scale: 1, useDefaultActor: false, id: gid }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const c = t.replace(/^for \(;;\);?/, '').trim();
          const json = JSON.parse(c.split('\n')[0]);
          const edges = json?.data?.node?.highlighted_units?.edges || [];
          const highlights = edges.map(e => ({ id: e.node?.id, storyId: e.node?.story?.id, message: e.node?.story?.message?.text || '', authorName: e.node?.story?.actors?.[0]?.name || 'Unknown', isPinned: true }));
          return { success: true, highlights, total: highlights.length };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  async MARK_ANNOUNCEMENT(data) {
    const { groupId, storyId } = data;
    if (!groupId || !storyId) return { success: false, error: 'Missing data' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [groupId, storyId],
        func: async (gid, sid) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '24968425266164144', variables: JSON.stringify({ input: { client_mutation_id: String(Date.now()), actor_id: userId, group_id: gid, story_id: sid }, highlighted_stories: [], privacySelectorRenderLocation: 'COMET_STREAM', scale: 1, renderLocation: 'group', useDefaultActor: true }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
          return json?.data ? { success: true } : { success: false, error: 'Failed' };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  async REMOVE_ANNOUNCEMENT(data) {
    const { storyId } = data;
    if (!storyId) return { success: false, error: 'Missing storyId' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    try {
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id }, world: 'MAIN', args: [storyId],
        func: async (sid) => {
          let userId, dtsg, lsd = '';
          if (typeof require === 'function') {
            try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
            try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
            try { lsd = require('LSD')?.token || ''; } catch(e) {}
          }
          if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
          if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
          const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '26131383586485490', variables: JSON.stringify({ input: { client_mutation_id: String(Date.now()), actor_id: userId, story_id: sid }, feedLocation: 'GROUP', feedbackSource: 0, focusCommentID: null, scale: 1, useDefaultActor: false, privacySelectorRenderLocation: 'COMET_STREAM', renderLocation: 'group' }) });
          const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
          const t = await r.text();
          const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
          return json?.data ? { success: true } : { success: false, error: 'Failed' };
        }
      });
      return results?.[0]?.result || { success: false, error: 'Script failed' };
    } catch (e) { return { success: false, error: e.message }; }
  },

  // === Delete Comments ===
  async DELETE_COMMENTS_ADMIN(data) {
    const { groupId, commentIds = [], delayMs = 1500 } = data;
    if (!groupId || !commentIds.length) return { success: false, error: 'Missing data' };
    const tab = await findFacebookTab();
    if (!tab) return { success: false, error: 'Open Facebook' };
    const results = { total: commentIds.length, success: 0, failed: 0, items: [] };
    for (let i = 0; i < commentIds.length; i++) {
      try {
        const r = await chrome.scripting.executeScript({
          target: { tabId: tab.id }, world: 'MAIN', args: [groupId, commentIds[i]],
          func: async (gid, cid) => {
            let userId, dtsg, lsd = '';
            if (typeof require === 'function') {
              try { userId = require('CurrentUserInitialData')?.USER_ID; } catch(e) {}
              try { dtsg = require('DTSGInitialData')?.token; } catch(e) {}
              try { lsd = require('LSD')?.token || ''; } catch(e) {}
            }
            if (!userId) { const m = document.cookie.match(/c_user=(\d+)/); if (m) userId = m[1]; }
            if (!userId || !dtsg) return { success: false, error: 'Not logged in' };
            const p = new URLSearchParams({ av: userId, __user: userId, __a: '1', fb_dtsg: dtsg, lsd, fb_api_caller_class: 'RelayModern', doc_id: '24071059422528616', variables: JSON.stringify({ input: { client_mutation_id: String(Date.now()), actor_id: userId, admin_notes: '', group_id: gid, selected_rules: [], send_warning: false, share_feedback: false, source: 'group_mall', story_id: cid }, profileID: userId }) });
            const r = await fetch('https://www.facebook.com/api/graphql/', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: p.toString(), credentials: 'include' });
            const t = await r.text();
            const json = JSON.parse(t.replace(/^for \(;;\);?/, '').trim().split('\n')[0]);
            return json?.data ? { success: true } : { success: false, error: json?.errors?.[0]?.message || 'Failed' };
          }
        });
        const result = r?.[0]?.result || { success: false, error: 'Script failed' };
        results.items.push({ commentId: commentIds[i], ...result });
        if (result.success) results.success++; else results.failed++;
      } catch(e) { results.items.push({ commentId: commentIds[i], success: false, error: e.message }); results.failed++; }
      broadcast({ type: 'DELETE_COMMENT_PROGRESS', data: { current: i + 1, total: commentIds.length } });
      if (i < commentIds.length - 1) await new Promise(r => setTimeout(r, delayMs));
    }
    return { success: true, ...results };
  }
};

// ==================== Handler Aliases (Backwards Compatibility) ====================
// Web pages may use different names than the handler — these aliases bridge that gap

handlers.GET_PENDING_POSTS = handlers.FETCH_PENDING_POSTS;
handlers.GET_INVITE_FRIENDS = handlers.SEARCH_INVITABLE_FRIENDS;
handlers.INVITE_FRIEND_TO_GROUP = handlers.INVITE_FRIENDS;
handlers.GET_GROUP_MEMBERS = handlers.FETCH_GROUP_MEMBERS;
handlers.BLOCK_GROUP_MEMBER = async (data) => {
  // Single member block adapter — web page sends { groupId, userId } for one user
  return handlers.BLOCK_MEMBERS({ groupId: data.groupId, userIds: [data.userId || data.memberId], options: data.options || {}, delayMs: 0 });
};
handlers.GET_GROUP_ADMINS_AND_MEMBERS = async (data) => {
  // Combined fetch for admin-manage page
  const [adminsResult, membersResult] = await Promise.all([
    handlers.FETCH_GROUP_ADMINS({ groupId: data.groupId }),
    handlers.FETCH_GROUP_MEMBERS({ groupId: data.groupId })
  ]);
  return {
    success: adminsResult.success || membersResult.success,
    admins: adminsResult.success ? adminsResult.admins : [],
    members: membersResult.success ? membersResult.members : [],
  };
};
handlers.ADD_GROUP_ADMIN = handlers.ADD_ADMIN;
handlers.REMOVE_GROUP_ADMIN = handlers.REMOVE_ADMIN;
handlers.GET_GROUP_POSTS = async (data) => {
  // Adapter: pages send groupId, but FETCH_GROUP_POSTS expects sourceGroupId
  return handlers.FETCH_GROUP_POSTS({ ...data, sourceGroupId: data.sourceGroupId || data.groupId });
};
handlers.DELETE_GROUP_POST = async (data) => {
  // Single post delete adapter
  return handlers.DELETE_POSTS_ADMIN({ groupId: data.groupId, storyIds: [data.postId || data.storyId], delayMs: 0 });
};
handlers.PIN_ANNOUNCEMENT = async (data) => {
  return handlers.MARK_ANNOUNCEMENT({ groupId: data.groupId, storyId: data.postId || data.storyId });
};
handlers.UNPIN_ANNOUNCEMENT = async (data) => {
  return handlers.REMOVE_ANNOUNCEMENT({ groupId: data.groupId, storyId: data.postId || data.storyId });
};
handlers.GET_GROUP_COMMENTS = async (data) => {
  // Adapter: pages send groupId, but FETCH_GROUP_POSTS expects sourceGroupId
  return handlers.FETCH_GROUP_POSTS({ ...data, sourceGroupId: data.sourceGroupId || data.groupId });
};
handlers.DELETE_COMMENT_AS_ADMIN = async (data) => {
  // Single comment delete adapter
  return handlers.DELETE_COMMENTS_ADMIN({ groupId: data.groupId, commentIds: [data.commentId], delayMs: 0 });
};
handlers.GET_SPAM_QUEUE = handlers.FETCH_SPAM_QUEUE;

// === Page-to-Handler aliases (invite, block, admin) ===
handlers.GET_INVITE_FRIENDS = handlers.SEARCH_INVITABLE_FRIENDS;
handlers.INVITE_FRIEND_TO_GROUP = handlers.INVITE_FRIENDS;
handlers.GET_GROUP_MEMBERS = handlers.FETCH_GROUP_MEMBERS;
handlers.ADD_GROUP_ADMIN = async (data) => {
  return handlers.ADD_ADMIN({ groupId: data.groupId, userId: data.userId || data.memberId, adminType: data.adminType || data.role || 'admin' });
};
handlers.REMOVE_GROUP_ADMIN = async (data) => {
  return handlers.REMOVE_ADMIN({ groupId: data.groupId, userId: data.userId || data.memberId, adminType: data.adminType || data.role || 'admin' });
};
handlers.BLOCK_GROUP_MEMBER = async (data) => {
  // Smart block adapter: pass through all options
  const options = {
    deletePosts: data.deletePosts || data.options?.deletePosts || false,
    deleteComments: data.deleteComments || data.options?.deleteComments || false,
    applyToOtherGroups: data.applyToOtherGroups || data.options?.applyToOtherGroups || false,
    applyToLater: data.applyToLater || data.options?.applyToLater || false,
    deleteInvites: data.deleteInvites || data.options?.deleteInvites || false,
  };
  const userIds = data.userIds || [data.userId || data.memberId];
  return handlers.BLOCK_MEMBERS({ groupId: data.groupId, userIds, options, delayMs: data.delayMs || 2000 });
};
handlers.GET_GROUP_ADMINS_AND_MEMBERS = async (data) => {
  // Combo adapter: fetch admins + members in parallel
  const [adminsRes, membersRes] = await Promise.all([
    handlers.FETCH_GROUP_ADMINS(data),
    handlers.FETCH_GROUP_MEMBERS(data),
  ]);
  return {
    success: adminsRes?.success || membersRes?.success,
    admins: adminsRes?.admins || [],
    members: membersRes?.members || [],
  };
};
// Adapter: page uses groupId, handler uses sourceGroupId
handlers.GET_GROUP_POSTS = async (data) => {
  return handlers.FETCH_GROUP_POSTS({ sourceGroupId: data.groupId || data.sourceGroupId, count: data.count || 20, skipImageFetch: data.skipImageFetch || false, cursor: data.cursor || null });
};
// Adapter: page sends single postId/storyId, handler expects storyIds array
handlers.DELETE_POST = async (data) => {
  return handlers.DELETE_POSTS_ADMIN({ groupId: data.groupId, storyIds: [data.storyId || data.postId], delayMs: 0 });
};
// Adapter: page sends single commentId, handler expects commentIds array
handlers.DELETE_COMMENT = async (data) => {
  return handlers.DELETE_COMMENTS_ADMIN({ groupId: data.groupId, commentIds: [data.commentId || data.feedbackId], delayMs: 0 });
};
// Pin / Unpin announcements
handlers.PIN_ANNOUNCEMENT = handlers.MARK_ANNOUNCEMENT;
handlers.UNPIN_ANNOUNCEMENT = async (data) => {
  return handlers.REMOVE_ANNOUNCEMENT({ storyId: data.storyId || data.postId });
};
// Stub: GET_GROUP_COMMENTS not yet implemented - return empty
handlers.GET_GROUP_COMMENTS = async () => {
  return { success: true, comments: [] };
};

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  const type = message.type;
  const handler = handlers[type];
  
  if (!handler) {
    sendResponse({ success: false, error: 'Unknown: ' + type });
    return false;
  }
  
  handler(message.data || message)
    .then(sendResponse)
    .catch(err => sendResponse({ success: false, error: err.message }));
  
  return true;
});

// ==================== External Message Router (Web <-> Extension) ====================
// Allows web dashboard to communicate with extension

const ALLOWED_ORIGINS = [
  'https://fbtoolkit.com',
  'https://www.fbtoolkit.com',
  'http://localhost:3000', // Dev
  'http://localhost:3001',
];

chrome.runtime.onMessageExternal.addListener((message, sender, sendResponse) => {
  // Verify origin
  const origin = sender.origin || sender.url;
  const isAllowed = ALLOWED_ORIGINS.some(o => origin?.startsWith(o));
  
  if (!isAllowed) {
    log('External message rejected from:', origin);
    sendResponse({ success: false, error: 'Origin not allowed' });
    return false;
  }
  
  const type = message.type;
  const handler = handlers[type];
  
  if (!handler) {
    sendResponse({ success: false, error: 'Unknown: ' + type });
    return false;
  }
  
  log('External request:', type, 'from', origin);
  
  handler(message.data || message)
    .then(sendResponse)
    .catch(err => sendResponse({ success: false, error: err.message }));
  
  return true;
});

// ==================== Lifecycle ====================

// Set this to your production URL
const WEB_DASHBOARD_URL = 'https://fbtoolkit.com/app';
const DEV_DASHBOARD_URL = 'http://localhost:3000/app';

chrome.action.onClicked.addListener(() => {
  // Open web dashboard with extension ID for messaging
  const extId = chrome.runtime.id;
  
  // Use dev URL if extension is in developer mode (unpacked)
  const isDev = !('update_url' in chrome.runtime.getManifest());
  const baseUrl = isDev ? DEV_DASHBOARD_URL : WEB_DASHBOARD_URL;
  
  chrome.tabs.create({ url: `${baseUrl}?ext=${extId}` });
});

chrome.runtime.onInstalled.addListener(() => log('Installed'));


// Load anonymous state
chrome.storage.local.get('fb_anonymous').then(r => {
  isAnonymousMode = !!r.fb_anonymous;
});

// ==================== Scheduled Posts Executor ====================

async function executeScheduledPost(schedId) {
  const stored = await chrome.storage.local.get('fb_scheduled_posts');
  const list = stored.fb_scheduled_posts || [];
  const entry = list.find(p => p.id === schedId);
  
  if (!entry || entry.status !== 'pending') {
    log(`Scheduled post ${schedId} not found or already executed`);
    return;
  }
  
  log(`Executing scheduled post: ${schedId}`);
  
  // Mark as executing
  entry.status = 'executing';
  await chrome.storage.local.set({ fb_scheduled_posts: list });
  
  try {
    const result = await handlers.POST_GROUPS({
      groupIds: entry.groupIds,
      content: entry.content,
      posts: entry.posts,
      imageData: entry.imageData,
      isAnonymous: entry.isAnonymous
    });
    
    entry.status = result.success ? 'completed' : 'failed';
    entry.result = result;
    entry.executedAt = Date.now();
  } catch (e) {
    entry.status = 'failed';
    entry.error = e.message;
    entry.executedAt = Date.now();
  }
  
  await chrome.storage.local.set({ fb_scheduled_posts: list });
  broadcast({ type: 'SCHEDULED_POST_EXECUTED', data: entry });
  log(`Scheduled post ${schedId}: ${entry.status}`);
}

chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name.startsWith('sched_')) {
    executeScheduledPost(alarm.name);
  }
});

// Auto-sync to web platform on startup (browser restart)
setTimeout(async () => {
  try {
    const result = await syncToWebPlatform(false, true);
    log('Startup sync:', result.success ? 'OK' : result.error);
  } catch (e) {
    log('Startup sync failed:', e.message);
  }
}, 2000);

log('v11.1 ready (Full Groups + Anonymous + Web Sync + Scheduling)');
